﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/cloudformation/model/ModuleInfo.h>
#include <aws/cloudformation/model/ResourceStatus.h>
#include <aws/cloudformation/model/StackResourceDriftInformation.h>
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace CloudFormation {
namespace Model {

/**
 * <p>Contains detailed information about the specified stack
 * resource.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cloudformation-2010-05-15/StackResourceDetail">AWS
 * API Reference</a></p>
 */
class StackResourceDetail {
 public:
  AWS_CLOUDFORMATION_API StackResourceDetail() = default;
  AWS_CLOUDFORMATION_API StackResourceDetail(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_CLOUDFORMATION_API StackResourceDetail& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_CLOUDFORMATION_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_CLOUDFORMATION_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The name associated with the stack.</p>
   */
  inline const Aws::String& GetStackName() const { return m_stackName; }
  inline bool StackNameHasBeenSet() const { return m_stackNameHasBeenSet; }
  template <typename StackNameT = Aws::String>
  void SetStackName(StackNameT&& value) {
    m_stackNameHasBeenSet = true;
    m_stackName = std::forward<StackNameT>(value);
  }
  template <typename StackNameT = Aws::String>
  StackResourceDetail& WithStackName(StackNameT&& value) {
    SetStackName(std::forward<StackNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique identifier of the stack.</p>
   */
  inline const Aws::String& GetStackId() const { return m_stackId; }
  inline bool StackIdHasBeenSet() const { return m_stackIdHasBeenSet; }
  template <typename StackIdT = Aws::String>
  void SetStackId(StackIdT&& value) {
    m_stackIdHasBeenSet = true;
    m_stackId = std::forward<StackIdT>(value);
  }
  template <typename StackIdT = Aws::String>
  StackResourceDetail& WithStackId(StackIdT&& value) {
    SetStackId(std::forward<StackIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The logical name of the resource specified in the template.</p>
   */
  inline const Aws::String& GetLogicalResourceId() const { return m_logicalResourceId; }
  inline bool LogicalResourceIdHasBeenSet() const { return m_logicalResourceIdHasBeenSet; }
  template <typename LogicalResourceIdT = Aws::String>
  void SetLogicalResourceId(LogicalResourceIdT&& value) {
    m_logicalResourceIdHasBeenSet = true;
    m_logicalResourceId = std::forward<LogicalResourceIdT>(value);
  }
  template <typename LogicalResourceIdT = Aws::String>
  StackResourceDetail& WithLogicalResourceId(LogicalResourceIdT&& value) {
    SetLogicalResourceId(std::forward<LogicalResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name or unique identifier that corresponds to a physical instance ID of a
   * resource supported by CloudFormation.</p>
   */
  inline const Aws::String& GetPhysicalResourceId() const { return m_physicalResourceId; }
  inline bool PhysicalResourceIdHasBeenSet() const { return m_physicalResourceIdHasBeenSet; }
  template <typename PhysicalResourceIdT = Aws::String>
  void SetPhysicalResourceId(PhysicalResourceIdT&& value) {
    m_physicalResourceIdHasBeenSet = true;
    m_physicalResourceId = std::forward<PhysicalResourceIdT>(value);
  }
  template <typename PhysicalResourceIdT = Aws::String>
  StackResourceDetail& WithPhysicalResourceId(PhysicalResourceIdT&& value) {
    SetPhysicalResourceId(std::forward<PhysicalResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Type of resource. For more information, see <a
   * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">Amazon
   * Web Services resource and property types reference</a> in the <i>CloudFormation
   * User Guide</i>.</p>
   */
  inline const Aws::String& GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  template <typename ResourceTypeT = Aws::String>
  void SetResourceType(ResourceTypeT&& value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = std::forward<ResourceTypeT>(value);
  }
  template <typename ResourceTypeT = Aws::String>
  StackResourceDetail& WithResourceType(ResourceTypeT&& value) {
    SetResourceType(std::forward<ResourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Time the status was updated.</p>
   */
  inline const Aws::Utils::DateTime& GetLastUpdatedTimestamp() const { return m_lastUpdatedTimestamp; }
  inline bool LastUpdatedTimestampHasBeenSet() const { return m_lastUpdatedTimestampHasBeenSet; }
  template <typename LastUpdatedTimestampT = Aws::Utils::DateTime>
  void SetLastUpdatedTimestamp(LastUpdatedTimestampT&& value) {
    m_lastUpdatedTimestampHasBeenSet = true;
    m_lastUpdatedTimestamp = std::forward<LastUpdatedTimestampT>(value);
  }
  template <typename LastUpdatedTimestampT = Aws::Utils::DateTime>
  StackResourceDetail& WithLastUpdatedTimestamp(LastUpdatedTimestampT&& value) {
    SetLastUpdatedTimestamp(std::forward<LastUpdatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Current status of the resource.</p>
   */
  inline ResourceStatus GetResourceStatus() const { return m_resourceStatus; }
  inline bool ResourceStatusHasBeenSet() const { return m_resourceStatusHasBeenSet; }
  inline void SetResourceStatus(ResourceStatus value) {
    m_resourceStatusHasBeenSet = true;
    m_resourceStatus = value;
  }
  inline StackResourceDetail& WithResourceStatus(ResourceStatus value) {
    SetResourceStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Success/failure message associated with the resource.</p>
   */
  inline const Aws::String& GetResourceStatusReason() const { return m_resourceStatusReason; }
  inline bool ResourceStatusReasonHasBeenSet() const { return m_resourceStatusReasonHasBeenSet; }
  template <typename ResourceStatusReasonT = Aws::String>
  void SetResourceStatusReason(ResourceStatusReasonT&& value) {
    m_resourceStatusReasonHasBeenSet = true;
    m_resourceStatusReason = std::forward<ResourceStatusReasonT>(value);
  }
  template <typename ResourceStatusReasonT = Aws::String>
  StackResourceDetail& WithResourceStatusReason(ResourceStatusReasonT&& value) {
    SetResourceStatusReason(std::forward<ResourceStatusReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>User defined description associated with the resource.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  StackResourceDetail& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The content of the <code>Metadata</code> attribute declared for the resource.
   * For more information, see <a
   * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-metadata.html">Metadata
   * attribute</a> in the <i>CloudFormation User Guide</i>.</p>
   */
  inline const Aws::String& GetMetadata() const { return m_metadata; }
  inline bool MetadataHasBeenSet() const { return m_metadataHasBeenSet; }
  template <typename MetadataT = Aws::String>
  void SetMetadata(MetadataT&& value) {
    m_metadataHasBeenSet = true;
    m_metadata = std::forward<MetadataT>(value);
  }
  template <typename MetadataT = Aws::String>
  StackResourceDetail& WithMetadata(MetadataT&& value) {
    SetMetadata(std::forward<MetadataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Information about whether the resource's actual configuration differs, or has
   * <i>drifted</i>, from its expected configuration, as defined in the stack
   * template and any values specified as template parameters. For more information,
   * see <a
   * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/using-cfn-stack-drift.html">Detect
   * unmanaged configuration changes to stacks and resources with drift
   * detection</a>.</p>
   */
  inline const StackResourceDriftInformation& GetDriftInformation() const { return m_driftInformation; }
  inline bool DriftInformationHasBeenSet() const { return m_driftInformationHasBeenSet; }
  template <typename DriftInformationT = StackResourceDriftInformation>
  void SetDriftInformation(DriftInformationT&& value) {
    m_driftInformationHasBeenSet = true;
    m_driftInformation = std::forward<DriftInformationT>(value);
  }
  template <typename DriftInformationT = StackResourceDriftInformation>
  StackResourceDetail& WithDriftInformation(DriftInformationT&& value) {
    SetDriftInformation(std::forward<DriftInformationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains information about the module from which the resource was created, if
   * the resource was created from a module included in the stack template.</p>
   */
  inline const ModuleInfo& GetModuleInfo() const { return m_moduleInfo; }
  inline bool ModuleInfoHasBeenSet() const { return m_moduleInfoHasBeenSet; }
  template <typename ModuleInfoT = ModuleInfo>
  void SetModuleInfo(ModuleInfoT&& value) {
    m_moduleInfoHasBeenSet = true;
    m_moduleInfo = std::forward<ModuleInfoT>(value);
  }
  template <typename ModuleInfoT = ModuleInfo>
  StackResourceDetail& WithModuleInfo(ModuleInfoT&& value) {
    SetModuleInfo(std::forward<ModuleInfoT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_stackName;

  Aws::String m_stackId;

  Aws::String m_logicalResourceId;

  Aws::String m_physicalResourceId;

  Aws::String m_resourceType;

  Aws::Utils::DateTime m_lastUpdatedTimestamp{};

  ResourceStatus m_resourceStatus{ResourceStatus::NOT_SET};

  Aws::String m_resourceStatusReason;

  Aws::String m_description;

  Aws::String m_metadata;

  StackResourceDriftInformation m_driftInformation;

  ModuleInfo m_moduleInfo;
  bool m_stackNameHasBeenSet = false;
  bool m_stackIdHasBeenSet = false;
  bool m_logicalResourceIdHasBeenSet = false;
  bool m_physicalResourceIdHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_lastUpdatedTimestampHasBeenSet = false;
  bool m_resourceStatusHasBeenSet = false;
  bool m_resourceStatusReasonHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_metadataHasBeenSet = false;
  bool m_driftInformationHasBeenSet = false;
  bool m_moduleInfoHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
