﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class DeleteSSHPublicKeyRequest : public IAMRequest {
 public:
  AWS_IAM_API DeleteSSHPublicKeyRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeleteSSHPublicKey"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the IAM user associated with the SSH public key.</p> <p>This
   * parameter allows (through its <a href="http://wikipedia.org/wiki/regex">regex
   * pattern</a>) a string of characters consisting of upper and lowercase
   * alphanumeric characters with no spaces. You can also include any of the
   * following characters: _+=,.@-</p>
   */
  inline const Aws::String& GetUserName() const { return m_userName; }
  inline bool UserNameHasBeenSet() const { return m_userNameHasBeenSet; }
  template <typename UserNameT = Aws::String>
  void SetUserName(UserNameT&& value) {
    m_userNameHasBeenSet = true;
    m_userName = std::forward<UserNameT>(value);
  }
  template <typename UserNameT = Aws::String>
  DeleteSSHPublicKeyRequest& WithUserName(UserNameT&& value) {
    SetUserName(std::forward<UserNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The unique identifier for the SSH public key.</p> <p>This parameter allows
   * (through its <a href="http://wikipedia.org/wiki/regex">regex pattern</a>) a
   * string of characters that can consist of any upper or lowercased letter or
   * digit.</p>
   */
  inline const Aws::String& GetSSHPublicKeyId() const { return m_sSHPublicKeyId; }
  inline bool SSHPublicKeyIdHasBeenSet() const { return m_sSHPublicKeyIdHasBeenSet; }
  template <typename SSHPublicKeyIdT = Aws::String>
  void SetSSHPublicKeyId(SSHPublicKeyIdT&& value) {
    m_sSHPublicKeyIdHasBeenSet = true;
    m_sSHPublicKeyId = std::forward<SSHPublicKeyIdT>(value);
  }
  template <typename SSHPublicKeyIdT = Aws::String>
  DeleteSSHPublicKeyRequest& WithSSHPublicKeyId(SSHPublicKeyIdT&& value) {
    SetSSHPublicKeyId(std::forward<SSHPublicKeyIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_userName;

  Aws::String m_sSHPublicKeyId;
  bool m_userNameHasBeenSet = false;
  bool m_sSHPublicKeyIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
