use strict;
use warnings;

package Printer::ESCPOS::Roles::Connection;

# PODNAME: Printer::ESCPOS::Roles::Connection
# ABSTRACT: Role for Connection Classes for Printer::ESCPOS
#
# This file is part of Printer-ESCPOS
#
# This software is copyright (c) 2015 by Shantanu Bhadoria.
#
# This is free software; you can redistribute it and/or modify it under
# the same terms as the Perl 5 programming language system itself.
#
our $VERSION = '0.003'; # TRIAL VERSION

# Dependencies

use 5.010;
use Moose::Role;

has _buffer => (
    is      => 'rw',
    default => '',
);


sub write {
    my ( $self, $raw ) = @_;

    $self->_buffer( $self->_buffer . $raw );
}


sub print {
    my ( $self, $raw ) = @_;
    my @chunks;
    my $buffer = $self->_buffer;
    my $n      = 64;               # Size of each chunk in bytes

    @chunks = unpack "a$n" x ( ( length($buffer) / $n ) - 1 ) . "a*", $buffer;
    for my $chunk (@chunks) {
        $self->_connection->write($chunk);
    }
    $self->_buffer('');
}

1;

__END__

=pod

=head1 NAME

Printer::ESCPOS::Roles::Connection - Role for Connection Classes for Printer::ESCPOS

=head1 VERSION

version 0.003

=head1 METHODS

=head2 write

Writes prepared data to the module buffer. This data is dispatched to printer with print() method. The print method takes care of buffer control issues.

=head2 print

Sends buffer data to the printer.

=head1 AUTHOR

Shantanu Bhadoria (shantanu@cpan.org)

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Shantanu Bhadoria.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
