#!/usr/bin/perl

use strict;
use warnings;

use Test::More tests => 3;
use File::Spec ();

use Test::Count::Filter ();
use IO::Scalar          ();

sub _slurp
{
    my $filename = shift;

    open my $in, '<', $filename
        or die "Cannot open '$filename' for slurping - $!";

    local $/;
    my $contents = <$in>;

    close($in);

    return $contents;
}

my $mytest = "T" . "E" . "S" . "T";
{
    open my $in, "<",
        File::Spec->catfile( "t", "sample-data", "test-scripts",
        "01-parser.pl" );
    my $buffer = "";
    my $out    = IO::Scalar->new( \$buffer );

    my $filter = Test::Count::Filter->new(
        {
            'input_fh'  => $in,
            'output_fh' => $out,
        }
    );

    $filter->process();

    # TEST
    is(
        $buffer,
        _slurp(
            File::Spec->catfile(
                "t", "sample-data", "test-scripts", "01-parser-good.pl"
            )
        ),
        "Testing for expected"
    );
}

{
    open my $in, "<",
        File::Spec->catfile( "t", "sample-data", "test-scripts", "basic.arc" );
    my $buffer = "";
    my $out    = IO::Scalar->new( \$buffer );

    my $filter = Test::Count::Filter->new(
        {
            'input_fh'            => $in,
            'output_fh'           => $out,
            'assert_prefix_regex' => qr{; TEST},
            'plan_prefix_regex'   => qr{\(plan\s+},
        }
    );

    $filter->process();

    # TEST
    is( $buffer,
        <<"EOF", "Testing for expected in Lisp Code with customisations" );
;;; basic.arc.t - test some basic arc features.
;;;
;;; This is an example for a Lisp/Scheme/Arc file that should be processed
;;; using Test-Count.
;;;
;;; This file is licensed under the MIT X11 License:
;;; http://www.opensource.org/licenses/mit-license.php
;;;
;;; (C) Copyright by Shlomi Fish, 2008

(load "arctap.arc")

(plan 18)

; TEST*3
(ok 1 "1 is a true value")

(ok (is 3 3) "3 is equal to 3")

(ok (is (+ 20 4) 24) "20+4 is equal to 24")

(with (x 20)
    (= x (+ x 4))
    ; TEST
    (ok (is x 24) "Adding 4 to x == 20 yields x == 24")
    (= x (- x 15))
    ; TEST
    (ok (is x 9) "Subtracting 15 to get 9"))

; TEST
(ok (is (* 3 6) 18) "3*6 == 18")

; TEST
(ok (> 5 3) "5 > 3")

; TEST
(ok (>= 5 3) "5 >= 3")

; TEST
(ok (>= 5 5) "5 >= 5")

; TEST
(ok (< 3 5) "3 < 5")

; TEST
(ok (<= 3 5) "3 <= 5")

; TEST
(ok (<= 3 3) "3 <= 3")

; TEST
(ok (not nil) "nil is false")

; TEST
(ok (not (not 1)) "!!1 is true")

; TEST
(ok (not (> 3 5)) "3 is not > 5")

; TEST
(ok (not (is 3 5)) "3 is not 5")

;---------------------------

; TEST+2
(ok (not (< 10 0)) "10 is not less than 0")
(ok (not (<= 5 4)) "5 is not leq than 4")
EOF
}

{
    open my $in, "<",
        File::Spec->catfile( "t", "sample-data", "test-scripts",
        "with-indented-plan.pl" );
    my $buffer = "";
    my $out    = IO::Scalar->new( \$buffer );

    my $filter = Test::Count::Filter->new(
        {
            'input_fh'  => $in,
            'output_fh' => $out,
        }
    );

    $filter->process();

    # TEST
    is( $buffer, <<"EOF", "Testing for expected" );
#!/usr/bin/perl

use strict;
use warnings;

use Test::More;

if (exists(\$ENV{TEST_ME}))
{
    plan tests => 2;
}
else
{
    plan skip_all => 'Skipping';
}

# $mytest
ok (1, 'One test');

# $mytest
ok (1, 'Second test');
EOF
}
