#!perl

use strict;
use warnings;
use Test::More tests => 28;

# only used for testing...
use Text::Diff;
use FreezeThaw qw(freeze);

BEGIN {
    use_ok( 'Parse::AccessLog' ) || print "Bail out!\n";
}

my $p = new_ok('Parse::AccessLog');

# nginx
my $log_line = q{127.0.0.1 - - [11/Jan/2013:17:31:36 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"};
my $log_line_unchomped = $log_line . "\n";

my $log_line_ipv6 = q{::1 - - [21/Jan/2013:17:55:52 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"};
my $log_line_ipv6_unchomped = $log_line_ipv6 . "\n";

my ($rec, $rec_w_newline, $v6rec);

# ========================================================================
# lines with/wo newline should be transparently processed - 2 tests
# ========================================================================
$rec = $p->parse($log_line);
$rec_w_newline = $p->parse($log_line_unchomped);
is( length(diff(\freeze($rec), \freeze($rec_w_newline))), 0,
    'newline transparently processed' );
undef $rec; undef $rec_w_newline;

$rec = $p->parse($log_line_ipv6);
$rec_w_newline = $p->parse($log_line_ipv6_unchomped);
is( length(diff(\freeze($rec), \freeze($rec_w_newline))), 0,
    'newline transparently processed' );
undef $rec; undef $rec_w_newline;


# ========================================================================
# parse() called as OBJECT method - 8 tests
# ========================================================================
$rec = $p->parse($log_line);
is($rec->{remote_addr}, '127.0.0.1', 'got remote_addr');
is($rec->{remote_user}, '-', 'got remote_user');
is($rec->{time_local}, '11/Jan/2013:17:31:36 -0600', 'got time_local');
is($rec->{request}, 'GET / HTTP/1.1', 'got request');
is($rec->{status}, 200, 'got HTTP status code');
is($rec->{bytes_sent}, 612, 'got bytes_sent');
is($rec->{referer}, '-', 'got referer');
is($rec->{user_agent},
    "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0",
    'got user-agent string');
undef $rec;


# ========================================================================
# IPv6 string
# ========================================================================
$v6rec = $p->parse($log_line_ipv6);
is($v6rec->{remote_addr}, '::1', 'parsed IPV6 remote_addr');
is($v6rec->{remote_user}, '-', 'got remote_user');
is($v6rec->{time_local}, '21/Jan/2013:17:55:52 -0600', 'got time_local');
is($v6rec->{request}, 'POST / HTTP/1.1', 'got request');
is($v6rec->{status}, 405, 'got HTTP status code');
is($v6rec->{bytes_sent}, 173, 'got bytes_sent');
is($v6rec->{referer}, '-', 'got referer');
is($v6rec->{user_agent}, 'PHP-SOAP/5.4.10', 'parsed user agent string');
undef $v6rec;

# ========================================================================
# parse() called as CLASS method - 8 tests
# parsing w/other data...
# ========================================================================
$rec = Parse::AccessLog->parse(q{127.0.0.1 - - [14/Jan/2013:19:16:21 +0100] "GET /info.php?=PHPE9568F35-D428-11d2-A769-00AA001ACF42 HTTP/1.1" 200 2158 "http://localhost:8008/info.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"});
is($rec->{remote_addr}, '127.0.0.1', 'got remote_addr');
is($rec->{remote_user}, '-', 'got remote_user');
is($rec->{time_local}, '14/Jan/2013:19:16:21 +0100', 'got time_local');
is($rec->{request}, 'GET /info.php?=PHPE9568F35-D428-11d2-A769-00AA001ACF42 HTTP/1.1', 'Got request string');
is($rec->{status}, 200, 'got HTTP status code');
is($rec->{bytes_sent}, 2158, 'got bytes_sent');
is($rec->{referer}, "http://localhost:8008/info.php", 'got referer');
is($rec->{user_agent},
    "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0",
    'got user-agent string');
undef $rec;

# remote_addr remote_user time_local request
# status bytes_sent referer user_agent

__END__
::1 - - [21/Jan/2013:17:55:52 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
127.0.0.1 - - [11/Jan/2013:17:31:36 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [11/Jan/2013:17:31:36 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [11/Jan/2013:17:31:47 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [11/Jan/2013:17:31:47 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [11/Jan/2013:17:31:47 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:18:39:24 +0100] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:18:39:24 +0100] "GET /favicon.ico HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:18:39:26 +0100] "GET /info.php HTTP/1.1" 200 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:18:58:19 +0100] "GET /info.php HTTP/1.1" 200 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:18:58:21 +0100] "GET /info.php HTTP/1.1" 200 5 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:58:25 +0100] "HEAD /info.php HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:06 +0100] "HEAD /info.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:10 +0100] "HEAD /info.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:13 +0100] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:16 +0100] "HEAD /infowars HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:21 +0100] "GET /infowars HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:18:59:24 +0100] "GET /infowars HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:33 +0100] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:34 +0100] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:38 +0100] "GET / HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:42 +0100] "GET /infowars HTTP/1.1" 404 169 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:44 +0100] "GET /infowars HTTP/1.1" 404 169 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:45 +0100] "GET /infowars HTTP/1.1" 404 169 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:00:52 +0100] "GET /info.php HTTP/1.1" 502 537 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:07:09 +0100] "GET /info.php HTTP/1.1" 200 5 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:10:16 +0100] "GET /info.php HTTP/1.1" 200 5 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:11:24 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:11:45 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:12:07 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:12:25 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:12:27 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:14:28 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:15:38 +0100] "GET /info.php HTTP/1.1" 404 36 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:16:06 +0100] "GET /info.php HTTP/1.1" 200 44733 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:16:12 +0100] "HEAD /info.php HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:16:20 +0100] "GET /info.phpl HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:16:21 +0100] "GET /info.php HTTP/1.1" 200 45116 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:16:21 +0100] "GET /info.php?=PHPE9568F34-D428-11d2-A769-00AA001ACF42 HTTP/1.1" 200 2536 "http://localhost:8008/info.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:16:21 +0100] "GET /info.php?=PHPE9568F35-D428-11d2-A769-00AA001ACF42 HTTP/1.1" 200 2158 "http://localhost:8008/info.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:16:27 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:19:18:16 +0100] "GET /wordpress HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:19:06 +0100] "GET /wordpress HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:19:08 +0100] "GET /wordpress HTTP/1.1" 500 193 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:19:19:53 +0100] "GET /wordpress HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:19:19:55 +0100] "HEAD /wordpress HTTP/1.1" 500 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:16:11 +0100] "GET /wordpress HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:16:14 +0100] "GET /wordpress/ HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:16:16 +0100] "GET / HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:16:18 +0100] "GET / HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:16:24 +0100] "GET /index.html HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:38:51 +0100] "GET / HTTP/1.1" 500 193 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:38:57 +0100] "GET /index.html HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:17 +0100] "GET /index.html HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:19 +0100] "GET / HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:22 +0100] "GET /one HTTP/1.1" 404 169 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:24 +0100] "GET /index.html HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:30 +0100] "GET /wordpress HTTP/1.1" 301 185 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:32 +0100] "GET /wordpress/ HTTP/1.1" 500 4570 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:34 +0100] "GET /wordpress/ HTTP/1.1" 500 4570 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:39:51 +0100] "GET / HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:54 +0100] "GET /wordpress HTTP/1.1" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:54 +0100] "GET /wordpress/ HTTP/1.1" 500 4564 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:56 +0100] "GET /wordpress/wp-admin/setup-config.php HTTP/1.1" 200 1488 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:56 +0100] "GET /wordpress/wp-admin/css/install.css?ver=3.5 HTTP/1.1" 200 3963 "http://localhost:8008/wordpress/wp-admin/setup-config.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:56 +0100] "GET /wordpress/wp-includes/css/buttons.css?ver=3.5 HTTP/1.1" 200 9593 "http://localhost:8008/wordpress/wp-admin/setup-config.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:56 +0100] "GET /wordpress/wp-admin/images/wordpress-logo.png?ver=20120216 HTTP/1.1" 200 2480 "http://localhost:8008/wordpress/wp-admin/css/install.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:39:59 +0100] "GET /wordpressclear HTTP/1.1" 404 169 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:40:01 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:40:27 +0100] "GET /wordpress/wp-admin/setup-config.php?step=1 HTTP/1.1" 200 1970 "http://localhost:8008/wordpress/wp-admin/setup-config.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:40:27 +0100] "GET /wordpress/wp-admin/css/install.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:40:27 +0100] "GET /wordpress/wp-includes/css/buttons.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:40:33 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:41:10 +0100] "POST /wordpress/wp-admin/setup-config.php?step=2 HTTP/1.1" 200 749 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:10 +0100] "GET /wordpress/wp-admin/css/install.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:10 +0100] "GET /wordpress/wp-includes/css/buttons.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-admin/install.php HTTP/1.1" 200 3588 "http://localhost:8008/wordpress/wp-admin/setup-config.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 200 7007 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-admin/css/install.min.css?ver=3.5 HTTP/1.1" 200 3321 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 200 93658 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-admin/js/password-strength-meter.min.js?ver=3.5 HTTP/1.1" 200 374 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:12 +0100] "GET /wordpress/wp-admin/js/user-profile.min.js?ver=3.5 HTTP/1.1" 200 1575 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:17 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "POST /wordpress/wp-admin/install.php?step=2 HTTP/1.1" 200 1711 "http://localhost:8008/wordpress/wp-admin/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "GET /wordpress/wp-admin/css/install.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "GET /wordpress/wp-admin/js/password-strength-meter.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "GET /wordpress/wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:54 +0100] "GET /wordpress/wp-admin/js/user-profile.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:57 +0100] "GET /wordpress/wp-login.php HTTP/1.1" 200 2439 "http://localhost:8008/wordpress/wp-admin/install.php?step=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:57 +0100] "GET /wordpress/wp-admin/css/wp-admin.min.css?ver=3.5 HTTP/1.1" 200 114555 "http://localhost:8008/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:57 +0100] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 200 36190 "http://localhost:8008/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:57 +0100] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:41:59 +0100] "POST /wordpress/wp-cron.php?doing_wp_cron=1358192519.8354959487915039062500 HTTP/1.0" 499 0 "-" "WordPress/3.5; http://localhost:8008/wordpress"
127.0.0.1 - - [14/Jan/2013:20:41:59 +0100] "POST /wordpress/wp-login.php HTTP/1.1" 302 5 "http://localhost:8008/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:00 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 200 4114 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/load-scripts.php?c=0&load[]=jquery,utils,json2&ver=3.5 HTTP/1.1" 200 98986 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/load-styles.php?c=0&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5 HTTP/1.1" 200 134480 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/ HTTP/1.1" 200 61976 "http://localhost:8008/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-includes/css/editor.min.css?ver=3.5 HTTP/1.1" 200 44688 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/load-styles.php?c=0&dir=ltr&load=media-views,wp-jquery-ui-dialog&ver=3.5 HTTP/1.1" 200 27154 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/load-scripts.php?c=0&load[]=admin-bar,hoverIntent,common,wp-ajax-response,jquery-color,wp-lists,quicktags,jquery-query,admin-comments,jquery-ui-core,jquery-&load[]=ui-widget,jquery-ui-mouse,jquery-ui-sortable,postbox,dashboard,customize-base,customize-loader,thickbox,plugin-install,underscor&load[]=e,shortcode,media-upload,backbone,media-models,plupload,plupload-html5,plupload-flash,plupload-silverlight,plupload-html4,wp-plu&load[]=pload,media-views,media-editor,word-count,jquery-ui-resizable,jquery-ui-draggable,jquery-ui-button,jquery-ui-position,jquery-ui-&load[]=dialog,wpdialogs,wplink,wpdialogs-popup&ver=3.5 HTTP/1.1" 200 364436 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/menu-shadow.png HTTP/1.1" 200 89 "http://localhost:8008/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/icons32.png?ver=20121105 HTTP/1.1" 200 12989 "http://localhost:8008/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-includes/images/admin-bar-sprite.png?d=20120830 HTTP/1.1" 200 2470 "http://localhost:8008/wordpress/wp-admin/load-styles.php?c=0&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/arrows.png HTTP/1.1" 200 243 "http://localhost:8008/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/menu.png?ver=20121105 HTTP/1.1" 200 9165 "http://localhost:8008/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/xit.gif HTTP/1.1" 200 182 "http://localhost:8008/wordpress/wp-admin/load-styles.php?c=0&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/welcome-icons.png HTTP/1.1" 200 2152 "http://localhost:8008/wordpress/wp-admin/load-styles.php?c=0&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/images/media-button.png HTTP/1.1" 200 323 "http://localhost:8008/wordpress/wp-includes/css/editor.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-includes/js/thickbox/loadingAnimation.gif HTTP/1.1" 200 15238 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=wp-compression-test&test=1&1358192521521 HTTP/1.1" 200 1126 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=wp-compression-test&test=2&1358192521731 HTTP/1.1" 200 584 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:01 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=wp-compression-test&test=yes&1358192521851 HTTP/1.1" 200 11 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:02 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_primary HTTP/1.1" 499 0 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:02 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_secondary HTTP/1.1" 499 0 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:02 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_incoming_links HTTP/1.1" 499 0 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-includes/css/editor.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/index.php HTTP/1.1" 200 63048 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5 HTTP/1.1" 200 31518 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=jquery,utils,json2&ver=3.5 HTTP/1.1" 200 38656 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "POST /wordpress/wp-cron.php?doing_wp_cron=1358192523.3355140686035156250000 HTTP/1.0" 499 0 "-" "WordPress/3.5; http://localhost:8008/wordpress"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=media-views,wp-jquery-ui-dialog&ver=3.5 HTTP/1.1" 200 6420 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:03 +0100] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=admin-bar,hoverIntent,common,wp-ajax-response,jquery-color,wp-lists,quicktags,jquery-query,admin-comments,jquery-ui-core,jquery-&load[]=ui-widget,jquery-ui-mouse,jquery-ui-sortable,postbox,dashboard,customize-base,customize-loader,thickbox,plugin-install,underscor&load[]=e,shortcode,media-upload,backbone,media-models,plupload,plupload-html5,plupload-flash,plupload-silverlight,plupload-html4,wp-plu&load[]=pload,media-views,media-editor,word-count,jquery-ui-resizable,jquery-ui-draggable,jquery-ui-button,jquery-ui-position,jquery-ui-&load[]=dialog,wpdialogs,wplink,wpdialogs-popup&ver=3.5 HTTP/1.1" 200 118879 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_plugins HTTP/1.1" 200 936 "http://localhost:8008/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/ HTTP/1.1" 200 14350 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 200 12774 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 200 35292 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 200 6173 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 200 863 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:05 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_plugins HTTP/1.1" 200 936 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:06 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_primary HTTP/1.1" 200 1810 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:07 +0100] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_secondary HTTP/1.1" 200 2350 "http://localhost:8008/wordpress/wp-admin/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/?page_id=2 HTTP/1.1" 200 16971 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 200 786 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:11 +0100] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:42:16 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:42:23 +0100] "HEAD /wordpress/ HTTP/1.0" 301 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:42:39 +0100] "HEAD /wordpress/ HTTP/1.0" 301 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:42:43 +0100] "HEAD /wordpress/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:42:45 +0100] "HEAD /wordpress/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:42:56 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.8.1.18) Gecko/20081112 Fedora/2.0.0.18-1.fc8 Firefox/2.0.0.18"
127.0.0.1 - - [14/Jan/2013:20:43:03 +0100] "HEAD /wordpress/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:20:43:58 +0100] "GET /wordpress/ HTTP/1.1" 200 14344 "http://localhost:8008/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:43:58 +0100] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:43:58 +0100] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:43:58 +0100] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:43:58 +0100] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:20:44:04 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:20:45:56 +0100] "HEAD /wordpress/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:21:36:35 +0100] "GET / HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [14/Jan/2013:21:36:44 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:44 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:44 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:44 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:45 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:45 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:45 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:45 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:45 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:46 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:46 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:46 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:46 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:46 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:47 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:47 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:47 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:47 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:47 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:48 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:48 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:48 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:48 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:48 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:49 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:49 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:49 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:49 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:49 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:50 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:50 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:50 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:50 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:50 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:51 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:51 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:51 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:51 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:51 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:52 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:52 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:52 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:52 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:52 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:53 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:53 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:53 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:53 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:53 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:36:54 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:21:37:29 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:30 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:30 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:30 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:30 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:30 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:31 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:31 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:31 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:31 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:31 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:32 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:32 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:32 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:32 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:32 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:33 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:33 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:33 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:33 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:33 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:34 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:34 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:34 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:34 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:34 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:35 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:35 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:35 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:35 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:35 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:36 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:36 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:36 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:36 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:36 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:37 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:37 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:37 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:37 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:37 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:38 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:38 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:38 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:38 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:38 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:39 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:39 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:39 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:21:37:39 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [14/Jan/2013:22:21:46 +0100] "GET /wordpress/ HTTP/1.1" 200 14350 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:46 +0100] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:46 +0100] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:46 +0100] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:46 +0100] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:47 +0100] "GET /wordpress/ HTTP/1.1" 200 14350 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:47 +0100] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:47 +0100] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:47 +0100] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:22:21:47 +0100] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost:8008/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [14/Jan/2013:23:01:11 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:11 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:11 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:11 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:12 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:12 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:12 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:12 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:12 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:13 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:13 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:13 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:13 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:13 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:14 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:14 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:14 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:14 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:14 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:15 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:15 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:15 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:15 +0100] "GET / HTTP/1.0" 200 612 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:15 +0100] "GET /wordpress/ HTTP/1.0" 301 0 "-" "http_load 12mar2006"
127.0.0.1 - - [14/Jan/2013:23:01:16 +0100] "-" 400 0 "-" "-"
127.0.0.1 - - [15/Jan/2013:01:07:00 +0100] "GET / HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [15/Jan/2013:01:07:04 +0100] "GET /wordpress HTTP/1.1" 301 185 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [15/Jan/2013:01:07:09 +0100] "GET /wordpress/ HTTP/1.1" 200 7199 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:14 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:18 -0600] "HEAD /wiki HTTP/1.1" 404 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:20 -0600] "HEAD /wiki/ HTTP/1.1" 404 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:26 -0600] "HEAD /mediawiki HTTP/1.1" 404 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:28 -0600] "HEAD /mediawiki/ HTTP/1.1" 404 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:54 -0600] "GET /wiki HTTP/1.1" 301 185 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:46:59 -0600] "HEAD /wiki HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:47:01 -0600] "HEAD /wiki/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [19/Jan/2013:12:47:05 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:05 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:05 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:07 -0600] "GET /wiki/ HTTP/1.1" 200 807 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:07 -0600] "GET /wiki/skins/common/images/mediawiki.png HTTP/1.1" 200 22589 "http://localhost/wiki/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/mw-config/index.php HTTP/1.1" 200 41786 "http://localhost/wiki/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/common/config.js HTTP/1.1" 200 3471 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/resources/jquery/jquery.js HTTP/1.1" 200 266882 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/vector/images/page-base.png HTTP/1.1" 200 67 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/vector/images/page-fade.png HTTP/1.1" 200 115 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/vector/images/bullet-icon.png HTTP/1.1" 200 107 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/vector/images/border.png HTTP/1.1" 200 69 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/vector/images/portal-break.png HTTP/1.1" 200 96 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:09 -0600] "GET /wiki/skins/common/images/help-question.gif HTTP/1.1" 200 126 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:12 -0600] "POST /wiki/mw-config/index.php?page=Language HTTP/1.1" 302 1215 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:12 -0600] "GET /wiki/mw-config/index.php?page=ExistingWiki HTTP/1.1" 302 1231 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:12 -0600] "GET /wiki/mw-config/index.php?page=Welcome HTTP/1.1" 200 6825 "http://localhost/wiki/mw-config/index.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:13 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Welcome" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:13 -0600] "GET /wiki/skins/common/images/bullet.gif HTTP/1.1" 200 50 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:13 -0600] "GET /wiki/skins/vector/images/external-link-ltr-icon.png HTTP/1.1" 200 143 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:36 -0600] "POST /wiki/mw-config/index.php?page=Welcome HTTP/1.1" 302 1211 "http://localhost/wiki/mw-config/index.php?page=Welcome" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:36 -0600] "GET /wiki/mw-config/index.php?page=DBConnect HTTP/1.1" 200 16104 "http://localhost/wiki/mw-config/index.php?page=Welcome" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:36 -0600] "GET /wiki/skins/common/images/info-32.png HTTP/1.1" 200 1892 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:47:36 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:10 -0600] "GET /wiki/skins/common/images/help-question-hover.gif HTTP/1.1" 200 1246 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:18 -0600] "POST /wiki/mw-config/index.php?page=DBConnect HTTP/1.1" 200 16579 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:18 -0600] "GET /wiki/skins/common/images/critical-32.png HTTP/1.1" 200 1763 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:18 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48084 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:52 -0600] "POST /wiki/mw-config/index.php?page=DBConnect HTTP/1.1" 302 1219 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:52 -0600] "GET /wiki/mw-config/index.php?page=Upgrade HTTP/1.1" 302 1211 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:52 -0600] "GET /wiki/mw-config/index.php?page=DBSettings HTTP/1.1" 200 8318 "http://localhost/wiki/mw-config/index.php?page=DBConnect" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:52 -0600] "GET /wiki/skins/common/images/warning-32.png HTTP/1.1" 200 1301 "http://localhost/wiki/mw-config/index.php?page=DBSettings" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:52 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=DBSettings" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:58 -0600] "POST /wiki/mw-config/index.php?page=DBSettings HTTP/1.1" 302 1223 "http://localhost/wiki/mw-config/index.php?page=DBSettings" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:58 -0600] "GET /wiki/mw-config/index.php?page=Name HTTP/1.1" 200 9307 "http://localhost/wiki/mw-config/index.php?page=DBSettings" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:58 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Name" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:49:58 -0600] "GET /wiki/skins/vector/images/lock-icon.png HTTP/1.1" 200 174 "http://localhost/wiki/mw-config/index.php?css=ltr" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:50:24 -0600] "POST /wiki/mw-config/index.php?page=Name HTTP/1.1" 302 1199 "http://localhost/wiki/mw-config/index.php?page=Name" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:50:24 -0600] "GET /wiki/mw-config/index.php?page=Options HTTP/1.1" 200 19269 "http://localhost/wiki/mw-config/index.php?page=Name" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:50:24 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Options" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:50:25 -0600] "GET /wiki/skins/common/config-cc.css HTTP/1.1" 200 798 "http://creativecommons.org/choose/?partner=MediaWiki&exit_url=http%3A%2F%2Flocalhost%2Fwiki%2Fmw-config%2Findex.php%3Fpage%3DOptions%26SubmitCC%3Dindeed%26config__LicenseCode%3Dcc%26config_wgRightsUrl%3D%255Blicense_url%255D%26config_wgRightsText%3D%255Blicense_name%255D%26config_wgRightsIcon%3D%255Blicense_button%255D&lang=en&stylesheet=http%3A%2F%2Flocalhost%2Fwiki%2Fskins%2Fcommon%2Fconfig-cc.css" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:06 -0600] "POST /wiki/mw-config/index.php?page=Options HTTP/1.1" 302 1211 "http://localhost/wiki/mw-config/index.php?page=Options" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:06 -0600] "GET /wiki/mw-config/index.php?page=Install HTTP/1.1" 200 3610 "http://localhost/wiki/mw-config/index.php?page=Options" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:06 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Install" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:10 -0600] "POST /wiki/mw-config/index.php?page=Install HTTP/1.1" 200 3538 "http://localhost/wiki/mw-config/index.php?page=Install" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:10 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Install" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "POST /wiki/mw-config/index.php?page=Install HTTP/1.1" 302 1211 "http://localhost/wiki/mw-config/index.php?page=Install" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "GET /wiki/mw-config/index.php?page=Complete HTTP/1.1" 200 4156 "http://localhost/wiki/mw-config/index.php?page=Install" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "GET /wiki/skins/common/images/tick-32.png HTTP/1.1" 200 1137 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "GET /wiki/skins/common/images/download-32.png HTTP/1.1" 200 1759 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:51:39 -0600] "GET /wiki/mw-config/index.php?localsettings=1 HTTP/1.1" 200 4545 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:24 -0600] "GET /wiki/mw-config/index.php?page=Complete HTTP/1.1" 200 4162 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:24 -0600] "GET /wiki/mw-config/index.php?css=ltr HTTP/1.1" 200 48077 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:24 -0600] "GET /wiki/mw-config/index.php?localsettings=1 HTTP/1.1" 200 4551 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:27 -0600] "GET /wiki/index.php HTTP/1.1" 301 5 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:27 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 13652 "http://localhost/wiki/mw-config/index.php?page=Complete" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:27 -0600] "GET /wiki/skins/common/images/poweredby_mediawiki_88x31.png HTTP/1.1" 200 3525 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 200 56956 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 200 449 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery%2Cmediawiki&only=scripts&skin=vector&version=20121204T210731Z HTTP/1.1" 200 162763 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185139Z&* HTTP/1.1" 200 26893 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/skins/common/images/wiki.png HTTP/1.1" 200 24358 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:28 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185139Z&* HTTP/1.1" 200 31515 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:33 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:12:53:41 -0600] "GET / HTTP/1.0" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:41 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:41 -0600] "GET /favicon.ico HTTP/1.0" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:41 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/ HTTP/1.0" 301 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/ HTTP/1.1" 301 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.0" 200 13712 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 13740 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/skins/common/images/poweredby_mediawiki_88x31.png HTTP/1.0" 200 3525 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/skins/common/images/poweredby_mediawiki_88x31.png HTTP/1.1" 200 3525 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery%2Cmediawiki&only=scripts&skin=vector&version=20121204T210731Z HTTP/1.0" 200 162575 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery%2Cmediawiki&only=scripts&skin=vector&version=20121204T210731Z HTTP/1.1" 200 162763 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185139Z&* HTTP/1.0" 200 26857 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185139Z&* HTTP/1.1" 200 26893 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/skins/common/images/wiki.png HTTP/1.0" 200 24358 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/skins/common/images/wiki.png HTTP/1.1" 200 24358 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185139Z&* HTTP/1.0" 200 31471 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:43 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185139Z&* HTTP/1.1" 200 31515 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:12:53:44 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:12:53:49 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/ HTTP/1.1" 301 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 13742 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 26900 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:13:45:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 31515 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:14:59 -0600] "GET /wiki/index.php?title=Special:UserLogin&returnto=Main+Page HTTP/1.1" 200 12423 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:00 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Special:UserLogin&returnto=Main+Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:00 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "http://localhost/wiki/index.php?title=Special:UserLogin&returnto=Main+Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:00 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Special:UserLogin&returnto=Main+Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:04 -0600] "POST /wiki/index.php?title=Special:UserLogin&action=submitlogin&type=login&returnto=Main+Page HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Special:UserLogin&returnto=Main+Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:04 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 15243 "http://localhost/wiki/index.php?title=Special:UserLogin&returnto=Main+Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:15:04 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.api.watch%7Cmediawiki.page.ready%7Cmediawiki.page.watch.ajax&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 34889 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/index.php?title=Main_Page&action=edit HTTP/1.1" 200 18632 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.byteLength%2CbyteLimit%2Cclient%2Ccookie%2CmwExtension%2CtextSelection%7Cmediawiki.action.edit%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 40617 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_bold.png HTTP/1.1" 200 250 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_italic.png HTTP/1.1" 200 250 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_link.png HTTP/1.1" 200 280 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_extlink.png HTTP/1.1" 200 435 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_headline.png HTTP/1.1" 200 440 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_image.png HTTP/1.1" 200 483 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_nowiki.png HTTP/1.1" 200 322 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_media.png HTTP/1.1" 200 728 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_hr.png HTTP/1.1" 200 222 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:16:35 -0600] "GET /wiki/skins/common/images/button_sig.png HTTP/1.1" 200 926 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:17:12 -0600] "POST /wiki/index.php?title=Main_Page&action=submit HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:17:12 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 15411 "http://localhost/wiki/index.php?title=Main_Page&action=edit" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:17:14 -0600] "GET /wiki/index.php?title=Vulnerabilties&action=edit&redlink=1 HTTP/1.1" 200 17719 "http://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:17:18 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:18:19:30 -0600] "POST /wiki/index.php?title=Vulnerabilties&action=submit HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&redlink=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:19:30 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.1" 200 15310 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&redlink=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:19:36 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:18:21:38 -0600] "GET /wiki/index.php?title=Vulnerabilties&action=edit&section=1 HTTP/1.1" 200 18692 "http://localhost/wiki/index.php?title=Vulnerabilties" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:38 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:38 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:38 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:38 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.api.watch%7Cmediawiki.page.ready%7Cmediawiki.page.watch.ajax&skin=vector&version=20130120T001504Z&* HTTP/1.1" 200 34889 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:55 -0600] "POST /wiki/index.php?title=Vulnerabilties&action=submit HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:21:55 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.1" 200 15427 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:15 -0600] "GET /wiki/index.php?title=Vulnerabilties&action=edit&section=1 HTTP/1.1" 200 18727 "http://localhost/wiki/index.php?title=Vulnerabilties" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:15 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:15 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 304 0 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:31 -0600] "POST /wiki/index.php?title=Vulnerabilties&action=submit HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:31 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.1" 200 15283 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:35 -0600] "GET /wiki/index.php?title=Vulnerabilties&action=edit&section=1 HTTP/1.1" 200 18731 "http://localhost/wiki/index.php?title=Vulnerabilties" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:37 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [19/Jan/2013:18:29:42 -0600] "POST /wiki/index.php?title=Vulnerabilties&action=submit HTTP/1.1" 302 5 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:42 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.1" 200 15027 "http://localhost/wiki/index.php?title=Vulnerabilties&action=edit&section=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [19/Jan/2013:18:29:47 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [20/Jan/2013:21:00:05 -0600] "GET / HTTP/1.0" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:05 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:05 -0600] "GET /favicon.ico HTTP/1.0" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:05 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/ HTTP/1.0" 301 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/ HTTP/1.1" 301 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.0" 200 13825 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 200 13846 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/skins/common/images/poweredby_mediawiki_88x31.png HTTP/1.0" 200 3525 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/skins/common/images/poweredby_mediawiki_88x31.png HTTP/1.1" 200 3525 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 200 449 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 200 56956 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery%2Cmediawiki&only=scripts&skin=vector&version=20121204T210731Z HTTP/1.0" 200 162575 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery%2Cmediawiki&only=scripts&skin=vector&version=20121204T210731Z HTTP/1.1" 200 162763 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185328Z&* HTTP/1.0" 200 26857 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:16 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.client%2Ccookie%2CmwExtension%7Cmediawiki.legacy.ajax%2Cwikibits%7Cmediawiki.notify%2Cutil%7Cmediawiki.page.startup&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 26901 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:17 -0600] "GET /wiki/skins/common/images/wiki.png HTTP/1.0" 200 24358 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:17 -0600] "GET /wiki/skins/common/images/wiki.png HTTP/1.1" 200 24358 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185328Z&* HTTP/1.0" 200 31471 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:17 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=jquery.autoEllipsis%2CcheckboxShiftClick%2ChighlightText%2CmakeCollapsible%2Cmw-jump%2Cplaceholder%2Csuggestions%7Cmediawiki.api%2CsearchSuggest%2Cuser%7Cmediawiki.page.ready&skin=vector&version=20130119T185328Z&* HTTP/1.1" 200 31515 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:20 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.0" 200 13484 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:20 -0600] "GET /wiki/index.php?title=Vulnerabilties HTTP/1.1" 200 13512 "https://localhost/wiki/index.php?title=Main_Page" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [20/Jan/2013:21:00:22 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [20/Jan/2013:23:03:13 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [20/Jan/2013:23:04:38 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [20/Jan/2013:23:05:43 -0600] "-" 400 0 "-" "-"
::1 - - [20/Jan/2013:23:05:58 -0600] "-" 400 0 "-" "-"
::1 - - [20/Jan/2013:23:07:46 -0600] "GET / HTTP/1.1" 200 612 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/ HTTP/1.0" 301 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/ HTTP/1.1" 301 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.0" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/index.php?title=Main_Page HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=mediawiki.legacy.commonPrint%2Cshared%7Cskins.vector&only=styles&skin=vector&* HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=skins.vector&only=scripts&skin=vector&* HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:09 -0600] "GET /wiki/load.php?debug=false&lang=en&modules=startup&only=scripts&skin=vector&* HTTP/1.1" 200 11207 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:07:58:13 -0600] "GET /wordpress HTTP/1.0" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:13 -0600] "GET /wordpress HTTP/1.1" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:13 -0600] "GET /wordpress/ HTTP/1.0" 301 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:07:58:13 -0600] "GET /wordpress/ HTTP/1.1" 301 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:41 -0600] "GET /wordpress HTTP/1.1" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:41 -0600] "POST /wordpress/wp-cron.php?doing_wp_cron=1358776781.4374411106109619140625 HTTP/1.0" 499 0 "-" "WordPress/3.5; http://localhost/wordpress"
::1 - - [21/Jan/2013:07:59:41 -0600] "GET /wordpress/ HTTP/1.1" 200 7074 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:41 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 200 35292 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:41 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 200 863 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:41 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:47 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:07:59:51 -0600] "GET /wordpress/?page_id=2 HTTP/1.1" 200 9545 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:51 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:51 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 200 786 "http://localhost/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:51 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:54 -0600] "GET /wordpress/ HTTP/1.1" 200 7080 "http://localhost/wordpress/?page_id=2" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:54 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:07:59:54 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:12 -0600] "GET /wordpress/ HTTP/1.1" 200 7080 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:12 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:12 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-admin HTTP/1.1" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-admin/ HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-login.php?redirect_to=http%3A%2F%2Flocalhost%2Fwordpress%2Fwp-admin%2F&reauth=1 HTTP/1.1" 200 2404 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-admin/css/wp-admin.min.css?ver=3.5 HTTP/1.1" 200 114555 "http://localhost/wordpress/wp-login.php?redirect_to=http%3A%2F%2Flocalhost%2Fwordpress%2Fwp-admin%2F&reauth=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 200 7007 "http://localhost/wordpress/wp-login.php?redirect_to=http%3A%2F%2Flocalhost%2Fwordpress%2Fwp-admin%2F&reauth=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 200 36190 "http://localhost/wordpress/wp-login.php?redirect_to=http%3A%2F%2Flocalhost%2Fwordpress%2Fwp-admin%2F&reauth=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:17 -0600] "GET /wordpress/wp-admin/images/wordpress-logo.png?ver=20120216 HTTP/1.1" 200 2480 "http://localhost/wordpress/wp-admin/css/wp-admin.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:20 -0600] "POST /wordpress/wp-login.php HTTP/1.1" 200 3321 "http://localhost/wordpress/wp-login.php?redirect_to=http%3A%2F%2Flocalhost%2Fwordpress%2Fwp-admin%2F&reauth=1" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:20 -0600] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:20 -0600] "GET /wordpress/wp-admin/css/wp-admin.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:20 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "POST /wordpress/wp-login.php HTTP/1.1" 302 5 "http://localhost/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/ HTTP/1.1" 200 60696 "http://localhost/wordpress/wp-login.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 200 4114 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-includes/css/editor.min.css?ver=3.5 HTTP/1.1" 200 44688 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5 HTTP/1.1" 200 31487 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=jquery,utils,json2&ver=3.5 HTTP/1.1" 200 38656 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=media-views,wp-jquery-ui-dialog&ver=3.5 HTTP/1.1" 200 6420 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=admin-bar,hoverIntent,common,wp-ajax-response,jquery-color,wp-lists,quicktags,jquery-query,admin-comments,jquery-ui-core,jquery-&load[]=ui-widget,jquery-ui-mouse,jquery-ui-sortable,postbox,dashboard,customize-base,customize-loader,thickbox,plugin-install,underscor&load[]=e,shortcode,media-upload,backbone,media-models,plupload,plupload-html5,plupload-flash,plupload-silverlight,plupload-html4,wp-plu&load[]=pload,media-views,media-editor,word-count,jquery-ui-resizable,jquery-ui-draggable,jquery-ui-button,jquery-ui-position,jquery-ui-&load[]=dialog,wpdialogs,wplink,wpdialogs-popup&ver=3.5 HTTP/1.1" 200 118879 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/menu-shadow.png HTTP/1.1" 200 89 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/menu.png?ver=20121105 HTTP/1.1" 200 9165 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-includes/images/admin-bar-sprite.png?d=20120830 HTTP/1.1" 200 2470 "http://localhost/wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/arrows.png HTTP/1.1" 200 243 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/icons32.png?ver=20121105 HTTP/1.1" 200 12989 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/xit.gif HTTP/1.1" 200 182 "http://localhost/wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/welcome-icons.png HTTP/1.1" 200 2152 "http://localhost/wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:52 -0600] "GET /wordpress/wp-admin/images/media-button.png HTTP/1.1" 200 323 "http://localhost/wordpress/wp-includes/css/editor.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:53 -0600] "GET /wordpress/wp-includes/js/thickbox/loadingAnimation.gif HTTP/1.1" 200 15238 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_primary HTTP/1.1" 200 1810 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_incoming_links HTTP/1.1" 200 2138 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_secondary HTTP/1.1" 200 2071 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?action=dashboard-widgets&widget=dashboard_plugins HTTP/1.1" 200 953 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:01:58 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:01:58 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:02:04 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:10 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:12 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:14 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:15 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:16 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:27 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:29 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:36 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:43 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:44 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:45 -0600] "POST /wordpress/wp-admin/admin-ajax.php HTTP/1.1" 200 11 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:54 -0600] "GET /wordpress/wp-admin/options-permalink.php HTTP/1.1" 200 22185 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:54 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:54 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=jquery,utils&ver=3.5 HTTP/1.1" 200 37374 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:02:54 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=admin-bar,hoverIntent,common&ver=3.5 HTTP/1.1" 200 5298 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:00 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:03:01 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:01 -0600] "POST /wordpress/wp-admin/options-permalink.php HTTP/1.1" 200 22289 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:05 -0600] "GET /wordpress/ HTTP/1.1" 200 14287 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:06 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 200 12774 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:06 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:06 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 200 6173 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:06 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/index.php/sample-page/ HTTP/1.1" 200 16836 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/index.php/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/index.php/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/index.php/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/index.php/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:09 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/index.php/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:14 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:03:17 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:17 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:17 -0600] "GET /wordpress/wp-admin/ HTTP/1.1" 200 67135 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:17 -0600] "GET /wordpress/wp-includes/css/editor.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:22 -0600] "GET /wordpress/wp-admin/options-permalink.php HTTP/1.1" 200 22207 "http://localhost/wordpress/wp-admin/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:22 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:23 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:03:32 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:32 -0600] "POST /wordpress/wp-admin/options-permalink.php HTTP/1.1" 200 22273 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:36 -0600] "GET /wordpress/ HTTP/1.1" 200 14147 "http://localhost/wordpress/wp-admin/options-permalink.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:36 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:36 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:36 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:36 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/sample-page/ HTTP/1.1" 200 16692 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:39 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:43 -0600] "GET /wordpress/ HTTP/1.1" 200 14147 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:43 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:43 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:43 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:43 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/sample-page/ HTTP/1.1" 200 16692 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:44 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:48 -0600] "GET /wordpress/2013/01/ HTTP/1.1" 200 14264 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:48 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/2013/01/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:48 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/2013/01/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:48 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/2013/01/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:48 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/2013/01/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:51 -0600] "GET /wordpress/category/uncategorized/ HTTP/1.1" 200 14731 "http://localhost/wordpress/2013/01/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:51 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/category/uncategorized/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:51 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/category/uncategorized/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:51 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/category/uncategorized/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:51 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/category/uncategorized/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:53 -0600] "GET /wordpress/ HTTP/1.1" 200 14147 "http://localhost/wordpress/category/uncategorized/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:53 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:53 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:53 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:03:53 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/sample-page/ HTTP/1.1" 200 16692 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:01 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 304 0 "http://localhost/wordpress/sample-page/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:08:04:07 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:04:07 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:04:07 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:08:04:10 -0600] "HEAD /wordpress/sample-page/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:08:04:12 -0600] "GET /wordpress/sample-page/ HTTP/1.1" 200 9555 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:13:12:56 -0600] "GET / HTTP/1.0" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:12:56 -0600] "GET / HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:13:12:56 -0600] "GET /favicon.ico HTTP/1.0" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:12:56 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:13:13:45 -0600] "HEAD /wordpress HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:13:13:47 -0600] "HEAD /wordpress/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:13:13:48 -0600] "HEAD /wordpress/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:13:20:14 -0600] "HEAD /wordpress/ HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
fe80::1 - - [21/Jan/2013:13:20:17 -0600] "GET / HTTP/1.0" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:17 -0600] "GET / HTTP/1.1" 200 612 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/ HTTP/1.0" 200 14120 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/ HTTP/1.1" 200 14141 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 200 12774 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/wp-content/themes/twentytwelve/style.css?ver=3.5 HTTP/1.1" 200 35292 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 200 6173 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:18 -0600] "GET /wordpress/wp-content/themes/twentytwelve/js/navigation.js?ver=1.0 HTTP/1.1" 200 863 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:19 -0600] "GET /wordpress/wp-includes/images/admin-bar-sprite.png?d=20120830 HTTP/1.1" 200 2470 "http://localhost/wordpress/wp-includes/css/admin-bar.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:23 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 200 4114 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 200 36190 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/load-styles.php?c=1&dir=ltr&load=admin-bar,wp-admin,buttons&ver=3.5 HTTP/1.1" 200 31518 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=jquery,utils,json2&ver=3.5 HTTP/1.1" 200 38656 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/images/menu-shadow.png HTTP/1.1" 200 89 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/images/menu.png?ver=20121105 HTTP/1.1" 200 9165 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/images/arrows.png HTTP/1.1" 200 243 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-content/themes/twentytwelve/screenshot.png HTTP/1.1" 200 171045 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-content/themes/p2/screenshot.png HTTP/1.1" 200 34032 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-content/themes/twentyeleven/screenshot.png HTTP/1.1" 200 197796 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/themes.php HTTP/1.1" 200 36640 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/load-scripts.php?c=1&load[]=admin-bar,hoverIntent,common,thickbox,theme,customize-base,customize-loader&ver=3.5 HTTP/1.1" 200 13384 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-admin/images/icons32.png?ver=20121105 HTTP/1.1" 200 12989 "http://localhost/wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /wordpress/wp-includes/js/thickbox/loadingAnimation.gif HTTP/1.1" 200 15238 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:26 -0600] "GET /favicon.ico HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:29 -0600] "GET /wordpress/wp-admin/themes.php?action=activate&template=p2&stylesheet=p2&_wpnonce=b3b41df09e HTTP/1.1" 302 5 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:29 -0600] "GET /wordpress/wp-admin/themes.php?activated=true HTTP/1.1" 200 37037 "http://localhost/wordpress/wp-admin/themes.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:29 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/themes.php?activated=true" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:29 -0600] "GET /wordpress/wp-admin/css/colors-fresh.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/wp-admin/themes.php?activated=true" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:31 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/style.css HTTP/1.1" 200 35847 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/css/admin-bar.min.css?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.css?ver=20121105 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/jquery.ui.autocomplete.css?ver=1.8.11 HTTP/1.1" 200 1546 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/jquery.js?ver=1.8.3 HTTP/1.1" 200 93658 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/jquery.color.min.js?ver=2.1.0 HTTP/1.1" 200 9311 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/utils.min.js?ver=3.5 HTTP/1.1" 200 1855 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/comment-reply.min.js?ver=3.5 HTTP/1.1" 200 786 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/jquery.scrollTo-min.js?ver=20120402 HTTP/1.1" 200 1995 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/wp-locale.js?ver=20110415 HTTP/1.1" 200 4167 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/ui/jquery.ui.core.min.js?ver=1.9.2 HTTP/1.1" 200 4693 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/spin.js?ver=20120704 HTTP/1.1" 200 8691 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/ui/jquery.ui.widget.min.js?ver=1.9.2 HTTP/1.1" 200 6759 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/ui/jquery.ui.position.min.js?ver=1.9.2 HTTP/1.1" 200 6645 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/p2.user.bundle.js?ver=20110405 HTTP/1.1" 200 47043 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/thickbox/thickbox.js?ver=3.1-20121105 HTTP/1.1" 200 12417 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/js/p2.js?ver=20120220 HTTP/1.1" 200 40559 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-admin/js/media-upload.min.js?ver=3.5 HTTP/1.1" 200 1902 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/shortcode.min.js?ver=3.5 HTTP/1.1" 200 2772 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/underscore.min.js?ver=1.4.0 HTTP/1.1" 200 13430 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/ HTTP/1.1" 200 45807 "http://localhost/wordpress/wp-admin/themes.php?activated=true" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/css/buttons.min.css?ver=3.5 HTTP/1.1" 200 7007 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/plupload.js?ver=1.5.4 HTTP/1.1" 200 11904 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/media-models.min.js?ver=3.5 HTTP/1.1" 200 10736 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/backbone.min.js?ver=0.9.2 HTTP/1.1" 200 16464 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/admin-bar.min.js?ver=3.5 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/css/media-views.min.css?ver=3.5 HTTP/1.1" 200 23789 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/plupload.flash.js?ver=1.5.4 HTTP/1.1" 200 5541 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/plupload.html5.js?ver=1.5.4 HTTP/1.1" 200 16719 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/json2.min.js?ver=2011-02-23 HTTP/1.1" 200 3377 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/plupload.html4.js?ver=1.5.4 HTTP/1.1" 200 4788 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/plupload.silverlight.js?ver=1.5.4 HTTP/1.1" 200 5610 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/ui/jquery.ui.sortable.min.js?ver=1.9.2 HTTP/1.1" 200 23582 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/jquery/ui/jquery.ui.mouse.min.js?ver=1.9.2 HTTP/1.1" 200 2864 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/plupload/wp-plupload.min.js?ver=3.5 HTTP/1.1" 200 4634 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/media-views.min.js?ver=3.5 HTTP/1.1" 200 59081 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-includes/js/media-editor.min.js?ver=3.5 HTTP/1.1" 200 8382 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/i/sidebar_back.gif HTTP/1.1" 200 866 "http://localhost/wordpress/wp-content/themes/p2/style.css" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:32 -0600] "GET /wordpress/wp-content/themes/p2/i/icons.png HTTP/1.1" 200 2385 "http://localhost/wordpress/wp-content/themes/p2/style.css" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:38 -0600] "POST /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=logged_in_out&_loggedin=1ab4266c94 HTTP/1.1" 200 19 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:45 -0600] "POST /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=logged_in_out&_loggedin=1ab4266c94 HTTP/1.1" 200 19 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:46 -0600] "POST /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=logged_in_out&_loggedin=1ab4266c94 HTTP/1.1" 200 19 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:46 -0600] "POST /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=logged_in_out&_loggedin=1ab4266c94 HTTP/1.1" 200 19 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:20:46 -0600] "POST /wordpress/wp-admin/admin-ajax.php?p2ajax=true HTTP/1.1" 200 11 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:13:20:46 -0600] "POST /wordpress/wp-cron.php?doing_wp_cron=1358796046.2651939392089843750000 HTTP/1.0" 499 0 "-" "WordPress/3.5; http://localhost/wordpress"
127.0.0.1 - - [21/Jan/2013:13:20:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:32&frontpage=1&vp=&vp[]=1&_=1358796046167 HTTP/1.1" 200 1682 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:21:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796062782 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:21:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796076171 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:21:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796092785 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:21:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796106175 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:22:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796122787 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:22:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796136178 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:22:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796152789 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:22:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796166180 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:23:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796182793 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:23:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796196178 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:23:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796212795 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:23:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796226181 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:24:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796242798 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:24:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796256184 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:24:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796272801 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:24:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796286187 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:25:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796302803 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:25:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796316190 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:25:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796332806 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:25:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796346193 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:26:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796362809 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:26:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796376196 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:26:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796392812 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:26:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796406198 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:27:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796422815 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:27:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796436201 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:27:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796452817 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:27:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796466203 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:28:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796482819 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:28:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796496205 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:28:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796512828 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:28:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796526208 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:29:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796542830 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:29:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796556210 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:29:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796572832 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:29:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796586213 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:30:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796602834 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:30:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796616214 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:30:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796632840 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:30:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796646216 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:31:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796662842 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:31:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796676218 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:31:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796692844 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:31:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796706220 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:32:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796722853 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:32:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796736221 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:32:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796752856 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:32:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796766227 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:33:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796782860 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:33:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796796230 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:33:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796812863 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:33:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796826232 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:34:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796842864 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:34:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796856234 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:34:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796872867 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:34:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796886236 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:35:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796902870 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:35:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796916239 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:35:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796932872 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:35:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796946241 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:36:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796962879 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:36:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358796976246 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:36:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358796992884 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:36:46 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358797006250 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:37:02 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358797022886 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:37:16 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358797036251 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:13:37:32 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358797052891 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:37:27 -0600] "POST /wordpress/wp-cron.php?doing_wp_cron=1358800647.9170589447021484375000 HTTP/1.0" 499 0 "-" "WordPress/3.5; http://localhost/wordpress"
127.0.0.1 - - [21/Jan/2013:14:37:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800647821 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:37:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800673695 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:37:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800677826 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:38:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800703700 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:38:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800707828 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:38:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800733705 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:38:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800737834 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:39:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800763710 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:39:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800767837 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:39:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800793718 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:39:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800797839 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:40:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800823721 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:40:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800827843 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:40:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800853726 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:40:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800857846 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:41:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800883731 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:41:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800887849 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:41:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800913734 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:41:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800917850 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:42:04 -0600] "GET /drupal HTTP/1.0" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:04 -0600] "GET /drupal HTTP/1.1" 404 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal HTTP/1.0" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal HTTP/1.1" 301 185 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/ HTTP/1.0" 302 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/ HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/install.php HTTP/1.0" 200 4395 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/install.php HTTP/1.1" 200 4414 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 200 78602 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 200 2974 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/themes/seven/logo.png HTTP/1.1" 200 3905 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 200 14544 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 200 5350 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 200 5117 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 200 961 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 200 2035 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 200 3711 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 200 811 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 200 2947 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 200 18468 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/misc/favicon.ico HTTP/1.1" 200 1150 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/themes/seven/images/task-item.png HTTP/1.1" 200 105 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:12 -0600] "GET /drupal/themes/seven/images/buttons.png HTTP/1.1" 200 786 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "POST /drupal/install.php HTTP/1.1" 302 5 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:14:42:14 -0600] "POST /drupal/install.php HTTP/1.0" 302 0 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/install.php?profile=standard HTTP/1.0" 200 4176 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/install.php?profile=standard HTTP/1.1" 200 4195 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:14 -0600] "GET /drupal/themes/seven/images/task-check.png HTTP/1.1" 200 261 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "POST /drupal/install.php?profile=standard HTTP/1.1" 302 5 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:42:17 -0600] "POST /drupal/install.php?profile=standard HTTP/1.0" 302 0 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6628 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6647 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:17 -0600] "GET /drupal/misc/message-16-error.png HTTP/1.1" 200 519 "http://localhost/drupal/modules/system/system.admin.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800943738 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800947854 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358800973741 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:42:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358800977857 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6131 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6144 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:01 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:06 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:14:43:07 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:14:43:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801003746 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801007857 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6320 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6339 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:38 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:43 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:14:43:44 -0600] "-" 400 0 "-" "-"
fe80::1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:49 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801033752 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:43:54 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:14:43:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801037860 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:14:44:06 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:14:44:08 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:12 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:16 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:17 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801063754 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801067868 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801093759 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:44:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801097872 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:45:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801123761 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:45:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801127874 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:45:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801153763 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:45:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801157876 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:46:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801183766 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:46:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801187879 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:46:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801213769 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:46:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801217876 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:47:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801243771 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:47:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801247882 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:47:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801273774 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:47:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801277886 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:48:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801303778 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:48:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801307892 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:48:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801333783 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:48:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801337893 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:49:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801363787 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:49:27 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801367897 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:49:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801393796 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:49:57 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801397900 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:50:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801423800 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:50:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801427902 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:50:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801453805 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:50:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801457906 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:51:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801483805 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:51:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801487906 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:51:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801513824 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:51:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801517909 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:52:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801543829 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:52:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801547913 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:52:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801573833 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:52:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801577915 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:53:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801603835 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:53:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801607917 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:53:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801633837 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:53:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801637919 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:54:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801663841 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:54:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801667923 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:54:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801693844 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:54:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801697926 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:55:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801723847 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:55:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801727927 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:55:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801753848 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:55:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801757930 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:56:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801783853 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:56:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801787934 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:56:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801813857 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:56:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801817936 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:57:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801843859 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:57:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801847937 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:57:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801873860 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:57:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801877939 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:58:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801903883 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:58:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801907941 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:58:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801933885 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:58:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801937943 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:59:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801963888 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:59:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801967944 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:59:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358801993889 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:14:59:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358801997947 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:00:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802023891 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:00:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802027949 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:00:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802053893 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:15:00:54 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
127.0.0.1 - - [21/Jan/2013:15:00:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802057950 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:01:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802083895 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:01:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802087951 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802113898 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:01:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802117955 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:02:23 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802143901 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:02:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802147956 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:02:53 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802173903 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:02:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802177958 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:03:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802203906 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:03:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802207961 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:03:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802233907 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:03:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802237962 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:04:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802263909 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:04:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802267963 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:04:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802293911 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:04:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802297965 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:05:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802323914 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:05:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802327967 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:05:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802353917 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:05:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802357969 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:06:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802383919 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:06:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802387973 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:06:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802413924 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:06:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802417976 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:07:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802443925 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:07:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802447978 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:07:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802473928 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:07:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802477981 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:08:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802503932 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:08:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802507983 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:08:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802533934 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:08:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802537987 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:09:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802563937 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:09:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802567990 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:09:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802593939 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:09:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802597993 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:10:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802623941 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:10:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802627996 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:10:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802653943 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:10:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802657999 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:11:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802683947 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:11:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802688002 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:11:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802713949 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:11:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802718004 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:12:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802743951 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:12:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802748008 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:12:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802773955 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:12:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802778010 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:13:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802803959 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:13:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802808012 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:13:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802833962 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:13:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802838015 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:14:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802863965 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:14:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802868018 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:14:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802893968 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:14:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802898020 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:15:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802923971 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:15:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802928022 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:15:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802953974 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:15:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802958024 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:16:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358802983978 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:16:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358802988027 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:16:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803013980 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:16:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803018029 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:17:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803043982 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:17:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803048031 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:17:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803073984 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:17:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803078033 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:18:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803103985 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:18:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803108034 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:18:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803133988 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:18:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803138037 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:19:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803163993 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:19:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803168040 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:19:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803193997 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:19:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803198042 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:20:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803223998 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:20:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803228045 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:20:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803254001 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:20:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803258048 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:21:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803284002 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:21:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803288050 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:21:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803314006 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:21:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803318053 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:22:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803344155 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:22:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803348201 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:22:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803374157 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:22:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803378202 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:23:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803404159 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:23:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803408203 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:23:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803434161 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:23:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803438205 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:24:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803464163 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:24:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803468207 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:24:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803494165 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:24:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803498209 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:25:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803524167 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:25:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803528211 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:25:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803554170 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:25:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803558212 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:26:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803584172 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:26:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803588214 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:26:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803614174 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:26:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803618217 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:27:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803644177 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:27:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803648219 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:27:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803674179 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:27:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803678221 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:28:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803704181 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:28:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803708223 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:28:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803734183 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:28:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803738225 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:29:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803764184 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:29:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803768227 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:29:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803794186 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:29:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803798230 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:30:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803824188 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:30:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803828231 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:30:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803854190 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:30:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803858234 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:31:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803884193 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:31:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803888236 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:31:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803914195 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:31:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803918239 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:32:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803944197 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:32:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803948240 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:32:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358803974199 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:32:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358803978242 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:33:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804004204 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:33:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804008247 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:33:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804034208 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:33:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804038249 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:34:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804064212 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:34:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804068252 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:34:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804094216 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:34:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804098254 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:35:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804124219 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:35:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804128256 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:35:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804154222 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:35:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804158259 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:36:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804184225 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:36:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804188262 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:36:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804214229 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:36:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804218265 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:37:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804244232 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:37:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804248262 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:37:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804274235 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:37:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804278265 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:38:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804304238 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:38:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804308291 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:38:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804334242 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:38:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804338294 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:39:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804364245 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:39:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804368295 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:39:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804394248 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:39:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804398298 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:40:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804424250 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:40:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804428302 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:40:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804454252 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:40:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804458304 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:41:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804484256 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:41:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804488306 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:41:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804514259 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:41:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804518309 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:42:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804544262 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:42:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804548311 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:42:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804574265 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:42:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804578314 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:43:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804604268 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:43:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804608317 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:43:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804634270 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:43:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804638319 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:44:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804664271 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:44:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804668322 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:44:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804694274 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:44:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804698325 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:45:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804724276 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:45:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804728328 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:45:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804754279 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:45:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804758331 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:46:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804784281 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:46:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804788333 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:46:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804814283 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:46:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804818335 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:47:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804844286 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:47:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804848338 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:47:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804874288 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:47:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804878341 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:48:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804904290 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:48:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804908344 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:48:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804934293 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:48:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804938346 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:49:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804964295 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:49:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804968350 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:49:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358804994298 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:49:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358804998352 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:50:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805024300 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:50:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805028354 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:50:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805054303 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:50:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805058357 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:51:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805084306 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:51:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805088359 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:51:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805114308 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:51:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805118361 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:52:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805144311 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:52:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805148363 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:52:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805174313 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:52:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805178366 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:53:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805204316 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:53:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805208369 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:53:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805234318 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:53:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805238371 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:54:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805264320 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:54:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805268373 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:54:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805294323 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:54:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805298375 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:55:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805324327 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:55:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805328379 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:55:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805354330 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:55:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805358381 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:56:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805384333 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:56:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805388383 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:56:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805414335 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:56:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805418387 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:57:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805444338 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:57:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805448389 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:57:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805474341 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:57:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805478391 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:58:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805504344 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:58:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805508394 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:58:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805534348 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:58:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805538397 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:59:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805564351 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:59:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805568399 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:59:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805594353 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:15:59:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805598402 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:00:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805624355 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:00:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805628404 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:00:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805654357 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:00:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805658407 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:01:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805684359 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:01:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805688409 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:01:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805714362 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:01:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805718412 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:02:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805744367 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:02:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805748415 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:02:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805774371 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:02:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805778417 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:03:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805804373 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:03:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805808420 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:03:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805834376 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:03:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805838424 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:04:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805864379 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:04:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805868426 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:04:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805894382 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:04:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805898428 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:05:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805924385 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:05:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805928430 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:05:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805954388 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:05:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805958432 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:06:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358805984391 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:06:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358805988435 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:06:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806014394 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:06:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806018438 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:07:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806044397 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:07:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806048440 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:07:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806074399 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:07:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806078442 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:08:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806104402 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:08:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806108444 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:08:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806134404 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:08:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806138446 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:09:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806164406 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:09:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806168449 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:09:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806194409 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:09:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806198451 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:10:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806224411 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:10:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806228453 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:10:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806254413 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:10:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806258456 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:11:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806284415 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:11:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806288458 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:11:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806314417 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:11:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806318461 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:12:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806344419 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:12:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806348463 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:12:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806374421 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:12:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806378466 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:13:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806404423 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:13:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806408469 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:13:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806434425 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:13:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806438471 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:14:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806464429 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:14:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806468473 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:14:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806494431 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:14:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806498475 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:15:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806524433 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:15:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806528477 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:15:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806554435 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:15:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806558479 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:16:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806584437 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:16:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806588480 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:16:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806614439 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:16:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806618484 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:17:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806644442 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:17:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806648486 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:17:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806674443 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:17:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806678488 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:18:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806704445 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:18:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806708490 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:18:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806734448 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:18:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806738492 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:19:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806764452 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:19:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806768496 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:19:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806794455 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:19:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806798499 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:20:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806824458 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:20:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806828501 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:20:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806854460 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:20:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806858505 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:21:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806884463 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:21:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806888507 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:21:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806914467 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:21:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806918510 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:22:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806944470 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:22:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806948512 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:22:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358806974474 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:22:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358806978514 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:23:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807004477 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:23:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807008516 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:23:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807034480 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:23:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807038520 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:24:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807064482 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:24:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807068522 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:24:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807094484 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:24:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807098524 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:25:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807124487 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:25:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807128526 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:25:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807154489 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:25:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807158529 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:26:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807184491 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:26:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807188531 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:26:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807214494 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:26:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807218534 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:27:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807244496 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:27:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807248536 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:27:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807274498 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:27:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807278538 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:28:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807304500 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:28:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807308540 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:28:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807334502 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:28:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807338543 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:29:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807364505 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:29:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807368545 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:29:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807394507 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:29:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807398549 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:30:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807424509 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:30:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807428551 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:30:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807454511 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:30:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807458552 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:31:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807484514 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:31:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807488554 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:31:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807514516 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:31:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807518556 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:32:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807544518 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:32:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807548558 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:32:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807574520 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:32:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807578560 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:33:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807604523 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:33:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807608562 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:33:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807634525 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:33:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807638565 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:34:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807664527 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:34:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807668568 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:34:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807694530 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:34:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807698569 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:35:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807724532 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:35:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807728571 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:35:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807754534 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:35:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807758573 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:36:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807784537 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:36:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807788575 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:36:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807814539 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:36:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807818577 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:37:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807844544 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:37:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807848574 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:37:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807874548 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:37:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807878577 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:38:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807904550 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:38:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807908581 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:38:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807934553 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:38:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807938585 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:39:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807964555 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:39:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807968588 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:39:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358807994558 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:39:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358807998590 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:40:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808024561 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:40:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808028593 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:40:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808054564 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:40:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808058597 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:41:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808084566 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:41:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808088600 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:41:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808114569 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:41:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808118603 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:42:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808144572 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:42:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808148607 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:42:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808174574 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:42:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808178609 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:43:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808204577 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:43:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808208612 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:43:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808234580 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:43:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808238614 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:44:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808264582 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:44:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808268617 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:44:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808294585 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:44:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808298619 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:24 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808324587 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:28 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808328621 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:51 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:54 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_comments&load_time=2013-01-21%2019:20:46&lcwidget=false&vp[]=1&vp[]=4&_=1358808354585 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:16:45:56 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:16:45:57 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:16:45:57 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:16:45:58 -0600] "GET /wordpress/wp-admin/admin-ajax.php?p2ajax=true&action=get_latest_posts&load_time=2013-01-21%2019:20:46&frontpage=1&vp=&vp[]=1&vp[]=4&_=1358808358623 HTTP/1.1" 304 0 "http://localhost/wordpress/" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:16:49:23 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/ HTTP/1.0" 302 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/ HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/install.php HTTP/1.0" 200 4395 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/install.php HTTP/1.1" 200 4414 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:56 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "POST /drupal/install.php HTTP/1.1" 302 5 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:28:59 -0600] "POST /drupal/install.php HTTP/1.0" 302 0 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/install.php?profile=standard HTTP/1.0" 200 4176 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/install.php?profile=standard HTTP/1.1" 200 4195 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:28:59 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "POST /drupal/install.php?profile=standard HTTP/1.1" 302 5 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:29:00 -0600] "POST /drupal/install.php?profile=standard HTTP/1.0" 302 0 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:00 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 5966 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 5985 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 200 78602 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 200 2974 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 200 14544 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 200 5117 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 200 5350 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 200 2947 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 200 961 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 200 2035 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 200 811 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 200 3711 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 200 18468 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/themes/seven/logo.png HTTP/1.1" 200 3905 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/themes/seven/images/task-check.png HTTP/1.1" 200 261 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/misc/message-16-error.png HTTP/1.1" 200 519 "http://localhost/drupal/modules/system/system.admin.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:05 -0600] "GET /drupal/themes/seven/images/task-item.png HTTP/1.1" 200 105 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:29:39 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:39 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:29:43 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:43 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:44 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:17:29:46 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:46 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:29:49 -0600] "GET /wordpress/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:29:49 -0600] "GET /wordpress/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:21 -0600] "GET /wordpress/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:21 -0600] "GET /wordpress/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:30:23 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:23 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:30:24 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:24 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:26 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:17:30:26 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:30:26 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:30:33 -0600] "HEAD /drupal/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:35 -0600] "HEAD / HTTP/1.1" 200 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:37 -0600] "HEAD /drupal HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:41 -0600] "HEAD /drupal/index.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:43 -0600] "HEAD /drupal/index.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:45 -0600] "HEAD /drupal/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:46 -0600] "HEAD /drupal HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:48 -0600] "HEAD /drupal HTTP/1.1" 301 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:30:49 -0600] "HEAD /drupal/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:17:32:09 -0600] "GET /drupal/ HTTP/1.0" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:32:09 -0600] "GET /drupal/ HTTP/1.1" 502 537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:32:12 -0600] "HEAD /drupal/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:32:14 -0600] "HEAD /drupal/ HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:17:32:14 -0600] "-" 400 0 "-" "-"
::1 - - [21/Jan/2013:17:32:20 -0600] "HEAD /drupal/index.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:32:30 -0600] "HEAD /drupal/index.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:32:34 -0600] "HEAD /drupal/index.php HTTP/1.1" 502 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:33:13 -0600] "HEAD /drupal/index.php HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:33:15 -0600] "HEAD /drupal/ HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
::1 - - [21/Jan/2013:17:33:17 -0600] "HEAD /drupal/ HTTP/1.1" 302 0 "-" "curl/7.21.4 (universal-apple-darwin11.0) libcurl/7.21.4 OpenSSL/0.9.8r zlib/1.2.5"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/ HTTP/1.0" 302 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/ HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/install.php HTTP/1.0" 200 4395 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/install.php HTTP/1.1" 200 4414 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:20 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "POST /drupal/install.php HTTP/1.1" 302 5 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:33:22 -0600] "POST /drupal/install.php HTTP/1.0" 302 0 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/install.php?profile=standard HTTP/1.0" 200 4176 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/install.php?profile=standard HTTP/1.1" 200 4189 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:22 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "POST /drupal/install.php?profile=standard HTTP/1.1" 302 5 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:33:23 -0600] "POST /drupal/install.php?profile=standard HTTP/1.0" 302 0 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6537 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6550 "https://localhost/drupal/install.php?profile=standard" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/form.js?v=7.19 HTTP/1.1" 200 2460 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/collapse.js?v=7.19 HTTP/1.1" 200 3323 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/states.js?v=7.19 HTTP/1.1" 200 17355 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:23 -0600] "GET /drupal/misc/menu-collapsed.png HTTP/1.1" 200 105 "http://localhost/drupal/modules/system/system.theme.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:33:25 -0600] "GET /drupal/misc/menu-expanded.png HTTP/1.1" 200 106 "http://localhost/drupal/modules/system/system.theme.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6556 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/form.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/states.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/misc/collapse.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:29 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:36:31 -0600] "GET /drupal/install.php?profile=standard HTTP/1.0" 200 4176 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:31 -0600] "GET /drupal/install.php?profile=standard HTTP/1.1" 200 4195 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php HTTP/1.0" 200 4395 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php HTTP/1.1" 200 4414 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php?profile=standard HTTP/1.0" 200 4176 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php?profile=standard HTTP/1.1" 200 4195 "https://localhost/drupal/install.php" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:32 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6550 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6550 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/form.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/states.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/misc/collapse.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:34 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 6537 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 6556 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 200 78602 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 200 2974 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 200 14544 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/states.js?v=7.19 HTTP/1.1" 200 17355 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/form.js?v=7.19 HTTP/1.1" 200 2460 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/collapse.js?v=7.19 HTTP/1.1" 200 3323 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 200 2035 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 200 5350 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 200 961 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 200 5117 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 200 811 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 200 3711 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 200 18468 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 200 2947 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/logo.png HTTP/1.1" 200 3905 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/images/task-check.png HTTP/1.1" 200 261 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/images/task-item.png HTTP/1.1" 200 105 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/misc/menu-collapsed.png HTTP/1.1" 200 105 "http://localhost/drupal/modules/system/system.theme.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:35 -0600] "GET /drupal/themes/seven/images/buttons.png HTTP/1.1" 200 786 "http://localhost/drupal/themes/seven/style.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:17:36:37 -0600] "GET /drupal/misc/menu-expanded.png HTTP/1.1" 200 106 "http://localhost/drupal/modules/system/system.theme.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
::1 - - [21/Jan/2013:17:55:52 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
::1 - - [21/Jan/2013:18:11:20 -0600] "POST / HTTP/1.1" 405 173 "-" "PHP-SOAP/5.4.10"
::1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.0" 200 9483 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/install.php?profile=standard&locale=en HTTP/1.1" 200 9503 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/form.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/states.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/misc/collapse.js?v=7.19 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:10 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:25:15 -0600] "-" 400 0 "-" "-"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "POST /drupal/install.php?profile=standard&locale=en HTTP/1.1" 302 5 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
fe80::1 - - [21/Jan/2013:18:26:06 -0600] "POST /drupal/install.php?profile=standard&locale=en HTTP/1.0" 302 0 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/install.php?profile=standard&locale=en&op=start&id=1 HTTP/1.0" 200 4381 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/install.php?profile=standard&locale=en&op=start&id=1 HTTP/1.1" 200 4394 "https://localhost/drupal/install.php?profile=standard&locale=en" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/jquery.js?v=1.4.4 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/jquery.once.js?v=1.2 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/drupal.js?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/progress.js?v=7.19 HTTP/1.1" 200 3112 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.theme.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/batch.js?1358814366 HTTP/1.1" 200 939 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.messages.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.menus.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/user/user.css?0 HTTP/1.1" 200 1827 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.base.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.admin.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/themes/seven/style.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/themes/seven/reset.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/modules/system/system.maintenance.css?0 HTTP/1.1" 304 0 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/progress.gif HTTP/1.1" 200 5872 "http://localhost/drupal/modules/system/system.theme.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "POST /drupal/install.php?profile=standard&locale=en&id=1&op=do HTTP/1.1" 302 5 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:06 -0600] "GET /drupal/misc/message-24-error.png HTTP/1.1" 200 733 "http://localhost/drupal/modules/system/system.messages.css?0" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:18:26:17 -0600] "GET /drupal/install.php?profile=standard&locale=en&id=1&op=finished HTTP/1.1" 200 170 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
127.0.0.1 - - [21/Jan/2013:19:20:50 -0600] "GET /sputnik HTTP/1.1" 403 169 "-" "Mozilla/5.0 (Macintosh; Intel Mac OS X 10.7; rv:18.0) Gecko/20100101 Firefox/18.0"
