use warnings;
use strict;

use Test::More tests => 712;

{
	package FakeUtcDateTime;
	use Date::ISO8601 0.000 qw(ymd_to_cjdn);
	my $rdn_epoch_cjdn = 1721425;
	sub new {
		my($class, $y, $mo, $d, $h, $mi, $s) = @_;
		return bless({
			rdn => ymd_to_cjdn($y, $mo, $d) - $rdn_epoch_cjdn,
			sod => 3600*$h + 60*$mi + $s,
		}, $class);
	}
	sub utc_rd_values { ($_[0]->{rdn}, $_[0]->{sod}, 0) }
}

require_ok "DateTime::TimeZone::SystemV";

my $tz;

sub try($$$$) {
	my($timespec, $is_dst, $offset, $abbrev) = @_;
	$timespec =~ /\A([0-9]{4})-([0-9]{2})-([0-9]{2})T
			([0-9]{2}):([0-9]{2}):([0-9]{2})Z\z/x or die;
	my $dt = FakeUtcDateTime->new("$1", "$2", "$3", "$4", "$5", "$6");
	is !!$tz->is_dst_for_datetime($dt), !!$is_dst, "is DST for $timespec";
	is $tz->offset_for_datetime($dt), $offset, "offset for $timespec";
	is $tz->short_name_for_datetime($dt), $abbrev, "abbrev for $timespec";
}

# zero offsets
foreach("GMT0", "GMT+0", "GMT-0") {
	$tz = DateTime::TimeZone::SystemV->new($_);
	try "2005-01-03T00:00:00Z", 0, +0, "GMT";
}
$tz = DateTime::TimeZone::SystemV->new("AAA1BBB");
try "2005-04-24T09:00:00Z", 1, +0, "BBB";

# constant offset
$tz = DateTime::TimeZone::SystemV->new("EST5");
try "2005-01-03T00:00:00Z", 0, -18000, "EST";
try "2005-03-03T06:00:00Z", 0, -18000, "EST";
try "2005-06-07T09:01:10Z", 0, -18000, "EST";
try "2005-09-20T12:00:00Z", 0, -18000, "EST";
try "2005-11-02T20:00:00Z", 0, -18000, "EST";
try "2005-12-29T23:59:59Z", 0, -18000, "EST";

# default DST rules
$tz = DateTime::TimeZone::SystemV->new("EST5EDT");
try "2005-01-03T00:00:00Z", 0, -18000, "EST";
try "2005-03-03T06:00:00Z", 0, -18000, "EST";
try "2005-04-24T04:00:00Z", 0, -18000, "EST";
try "2005-04-24T04:59:59Z", 0, -18000, "EST";
try "2005-04-24T05:00:00Z", 0, -18000, "EST";
try "2005-04-24T05:59:59Z", 0, -18000, "EST";
try "2005-04-24T06:00:00Z", 0, -18000, "EST";
try "2005-04-24T06:59:59Z", 0, -18000, "EST";
try "2005-04-24T07:00:00Z", 1, -14400, "EDT";
try "2005-04-24T07:59:59Z", 1, -14400, "EDT";
try "2005-04-24T08:00:00Z", 1, -14400, "EDT";
try "2005-04-24T08:59:59Z", 1, -14400, "EDT";
try "2005-04-24T09:00:00Z", 1, -14400, "EDT";
try "2005-04-24T09:59:59Z", 1, -14400, "EDT";
try "2005-04-24T23:59:59Z", 1, -14400, "EDT";
try "2005-04-25T00:00:00Z", 1, -14400, "EDT";
try "2005-04-25T03:59:59Z", 1, -14400, "EDT";
try "2005-04-25T04:00:00Z", 1, -14400, "EDT";
try "2005-04-25T04:59:59Z", 1, -14400, "EDT";
try "2005-04-25T05:00:00Z", 1, -14400, "EDT";
try "2005-06-07T09:01:10Z", 1, -14400, "EDT";
try "2005-09-20T12:00:00Z", 1, -14400, "EDT";
try "2005-10-30T04:00:00Z", 1, -14400, "EDT";
try "2005-10-30T04:59:59Z", 1, -14400, "EDT";
try "2005-10-30T05:00:00Z", 1, -14400, "EDT";
try "2005-10-30T05:59:59Z", 1, -14400, "EDT";
try "2005-10-30T06:00:00Z", 0, -18000, "EST";
try "2005-10-30T06:59:59Z", 0, -18000, "EST";
try "2005-10-30T07:00:00Z", 0, -18000, "EST";
try "2005-10-30T07:59:59Z", 0, -18000, "EST";
try "2005-10-30T08:00:00Z", 0, -18000, "EST";
try "2005-10-30T08:59:59Z", 0, -18000, "EST";
try "2005-10-30T09:00:00Z", 0, -18000, "EST";
try "2005-10-30T09:59:59Z", 0, -18000, "EST";
try "2005-10-30T23:59:59Z", 0, -18000, "EST";
try "2005-10-31T00:00:00Z", 0, -18000, "EST";
try "2005-10-31T03:59:59Z", 0, -18000, "EST";
try "2005-10-31T04:00:00Z", 0, -18000, "EST";
try "2005-10-31T04:59:59Z", 0, -18000, "EST";
try "2005-10-31T05:00:00Z", 0, -18000, "EST";
try "2005-11-02T20:00:00Z", 0, -18000, "EST";
try "2005-12-29T23:59:59Z", 0, -18000, "EST";

# complex offsets and change times, abbreviation quoting, and "J" rules
$tz = DateTime::TimeZone::SystemV->new("<+A11>-5:30:45XYZ-6,J59/3:20,J60/9");
try "2004-01-03T00:00:00Z", 0, +19845, "+A11";
try "2004-02-27T00:00:00Z", 0, +19845, "+A11";
try "2004-02-27T21:30:00Z", 0, +19845, "+A11";
try "2004-02-27T21:49:14Z", 0, +19845, "+A11";
try "2004-02-27T21:49:15Z", 1, +21600, "XYZ";
try "2004-02-28T02:59:59Z", 1, +21600, "XYZ";
try "2004-02-28T03:00:00Z", 1, +21600, "XYZ";
try "2004-02-29T02:59:59Z", 1, +21600, "XYZ";
try "2004-02-29T03:00:00Z", 1, +21600, "XYZ";
try "2004-03-01T02:59:59Z", 1, +21600, "XYZ";
try "2004-03-01T03:00:00Z", 0, +19845, "+A11";
try "2004-03-02T02:59:59Z", 0, +19845, "+A11";
try "2004-03-02T03:00:00Z", 0, +19845, "+A11";
try "2004-12-29T23:59:59Z", 0, +19845, "+A11";
try "2005-01-03T00:00:00Z", 0, +19845, "+A11";
try "2005-02-27T00:00:00Z", 0, +19845, "+A11";
try "2005-02-27T21:30:00Z", 0, +19845, "+A11";
try "2005-02-27T21:49:14Z", 0, +19845, "+A11";
try "2005-02-27T21:49:15Z", 1, +21600, "XYZ";
try "2005-02-28T02:59:59Z", 1, +21600, "XYZ";
try "2005-02-28T03:00:00Z", 1, +21600, "XYZ";
try "2005-03-01T02:59:59Z", 1, +21600, "XYZ";
try "2005-03-01T03:00:00Z", 0, +19845, "+A11";
try "2005-03-02T02:59:59Z", 0, +19845, "+A11";
try "2005-03-02T03:00:00Z", 0, +19845, "+A11";
try "2005-12-29T23:59:59Z", 0, +19845, "+A11";

# "0" rules
$tz = DateTime::TimeZone::SystemV->new("<+A11>-5:30:45XYZ-6,58/3:20,59/9");
try "2004-01-03T00:00:00Z", 0, +19845, "+A11";
try "2004-02-27T00:00:00Z", 0, +19845, "+A11";
try "2004-02-27T21:30:00Z", 0, +19845, "+A11";
try "2004-02-27T21:49:14Z", 0, +19845, "+A11";
try "2004-02-27T21:49:15Z", 1, +21600, "XYZ";
try "2004-02-28T02:59:59Z", 1, +21600, "XYZ";
try "2004-02-28T03:00:00Z", 1, +21600, "XYZ";
try "2004-02-29T02:59:59Z", 1, +21600, "XYZ";
try "2004-02-29T03:00:00Z", 0, +19845, "+A11";
try "2004-03-01T02:59:59Z", 0, +19845, "+A11";
try "2004-03-01T03:00:00Z", 0, +19845, "+A11";
try "2004-03-02T02:59:59Z", 0, +19845, "+A11";
try "2004-03-02T03:00:00Z", 0, +19845, "+A11";
try "2004-12-29T23:59:59Z", 0, +19845, "+A11";
try "2005-01-03T00:00:00Z", 0, +19845, "+A11";
try "2005-02-27T00:00:00Z", 0, +19845, "+A11";
try "2005-02-27T21:30:00Z", 0, +19845, "+A11";
try "2005-02-27T21:49:14Z", 0, +19845, "+A11";
try "2005-02-27T21:49:15Z", 1, +21600, "XYZ";
try "2005-02-28T02:59:59Z", 1, +21600, "XYZ";
try "2005-02-28T03:00:00Z", 1, +21600, "XYZ";
try "2005-03-01T02:59:59Z", 1, +21600, "XYZ";
try "2005-03-01T03:00:00Z", 0, +19845, "+A11";
try "2005-03-02T02:59:59Z", 0, +19845, "+A11";
try "2005-03-02T03:00:00Z", 0, +19845, "+A11";
try "2005-12-29T23:59:59Z", 0, +19845, "+A11";

# "+" in offset, and "M" rules
$tz = DateTime::TimeZone::SystemV->new("AAA+3BBB,M3.2.3,M11.1.6");
try "2005-01-03T00:00:00Z", 0, -10800, "AAA";
try "2005-03-09T04:59:59Z", 0, -10800, "AAA";
try "2005-03-09T05:00:00Z", 1, -7200, "BBB";
try "2005-11-05T03:59:59Z", 1, -7200, "BBB";
try "2005-11-05T04:00:00Z", 0, -10800, "AAA";
try "2005-12-29T23:59:59Z", 0, -10800, "AAA";

# southern hemisphere DST
$tz = DateTime::TimeZone::SystemV->new("AAA3BBB,M11.1.6,M3.2.3");
try "2005-01-03T00:00:00Z", 1, -7200, "BBB";
try "2005-03-09T03:59:59Z", 1, -7200, "BBB";
try "2005-03-09T04:00:00Z", 0, -10800, "AAA";
try "2005-11-05T04:59:59Z", 0, -10800, "AAA";
try "2005-11-05T05:00:00Z", 1, -7200, "BBB";
try "2005-12-29T23:59:59Z", 1, -7200, "BBB";

# DST offset lower than standard offset
$tz = DateTime::TimeZone::SystemV->new("BBB2AAA3,M3.2.3,M11.1.6");
try "2005-01-03T00:00:00Z", 0, -7200, "BBB";
try "2005-03-09T03:59:59Z", 0, -7200, "BBB";
try "2005-03-09T04:00:00Z", 1, -10800, "AAA";
try "2005-11-05T04:59:59Z", 1, -10800, "AAA";
try "2005-11-05T05:00:00Z", 0, -7200, "BBB";
try "2005-12-29T23:59:59Z", 0, -7200, "BBB";

# DST change in following day
$tz = DateTime::TimeZone::SystemV->new("AAA-2BBB,M3.5.4/24:30,M10.5.5/24:30");
try "2012-03-28T10:00:00Z", 0, 7200, "AAA";
try "2012-03-29T10:00:00Z", 0, 7200, "AAA";
try "2012-03-29T21:59:59Z", 0, 7200, "AAA";
try "2012-03-29T22:00:00Z", 0, 7200, "AAA";
try "2012-03-29T22:29:59Z", 0, 7200, "AAA";
try "2012-03-29T22:30:00Z", 1, 10800, "BBB";
try "2012-03-30T10:00:00Z", 1, 10800, "BBB";
try "2012-03-31T10:00:00Z", 1, 10800, "BBB";
try "2012-10-25T09:00:00Z", 1, 10800, "BBB";
try "2012-10-26T09:00:00Z", 1, 10800, "BBB";
try "2012-10-26T20:59:59Z", 1, 10800, "BBB";
try "2012-10-26T21:00:00Z", 1, 10800, "BBB";
try "2012-10-26T21:29:59Z", 1, 10800, "BBB";
try "2012-10-26T21:30:00Z", 0, 7200, "AAA";
try "2012-10-27T09:00:00Z", 0, 7200, "AAA";
try "2012-10-28T09:00:00Z", 0, 7200, "AAA";

# change forward at end of year occurs in following UT year
$tz = DateTime::TimeZone::SystemV->new("AAA24BBB-24,J365/12,J65/12");
try "2005-01-31T00:00:00Z", 1, +86400, "BBB";
try "2005-03-03T00:00:00Z", 1, +86400, "BBB";
try "2005-03-04T00:00:00Z", 1, +86400, "BBB";
try "2005-03-05T00:00:00Z", 1, +86400, "BBB";
try "2005-03-05T11:59:59Z", 1, +86400, "BBB";
try "2005-03-05T12:00:00Z", 0, -86400, "AAA";
try "2005-03-06T00:00:00Z", 0, -86400, "AAA";
try "2005-03-07T00:00:00Z", 0, -86400, "AAA";
try "2005-03-08T00:00:00Z", 0, -86400, "AAA";
try "2005-12-29T00:00:00Z", 0, -86400, "AAA";
try "2005-12-29T23:59:59Z", 0, -86400, "AAA";
try "2005-12-30T00:00:00Z", 0, -86400, "AAA";
try "2005-12-30T23:59:59Z", 0, -86400, "AAA";
try "2005-12-31T00:00:00Z", 0, -86400, "AAA";
try "2005-12-31T23:59:59Z", 0, -86400, "AAA";
try "2006-01-01T00:00:00Z", 0, -86400, "AAA";
try "2006-01-01T11:59:59Z", 0, -86400, "AAA";
try "2006-01-01T12:00:00Z", 1, +86400, "BBB";
try "2006-01-01T23:59:59Z", 1, +86400, "BBB";
try "2006-01-02T00:00:00Z", 1, +86400, "BBB";
try "2006-01-02T23:59:59Z", 1, +86400, "BBB";
try "2006-01-03T00:00:00Z", 1, +86400, "BBB";
try "2006-01-03T23:59:59Z", 1, +86400, "BBB";

# change back at start of year occurs in preceding UT year
$tz = DateTime::TimeZone::SystemV->new("AAA24BBB-24,J65/12,J1/12");
try "2005-01-31T00:00:00Z", 0, -86400, "AAA";
try "2005-03-05T00:00:00Z", 0, -86400, "AAA";
try "2005-03-06T00:00:00Z", 0, -86400, "AAA";
try "2005-03-07T00:00:00Z", 0, -86400, "AAA";
try "2005-03-07T11:59:59Z", 0, -86400, "AAA";
try "2005-03-07T12:00:00Z", 1, +86400, "BBB";
try "2005-03-08T00:00:00Z", 1, +86400, "BBB";
try "2005-03-09T00:00:00Z", 1, +86400, "BBB";
try "2005-03-10T00:00:00Z", 1, +86400, "BBB";
try "2005-12-28T00:00:00Z", 1, +86400, "BBB";
try "2005-12-28T23:59:59Z", 1, +86400, "BBB";
try "2005-12-29T00:00:00Z", 1, +86400, "BBB";
try "2005-12-29T23:59:59Z", 1, +86400, "BBB";
try "2005-12-30T00:00:00Z", 1, +86400, "BBB";
try "2005-12-30T23:59:59Z", 1, +86400, "BBB";
try "2005-12-31T00:00:00Z", 1, +86400, "BBB";
try "2005-12-31T11:59:59Z", 1, +86400, "BBB";
try "2005-12-31T12:00:00Z", 0, -86400, "AAA";
try "2005-12-31T23:59:59Z", 0, -86400, "AAA";
try "2006-01-01T00:00:00Z", 0, -86400, "AAA";
try "2006-01-01T23:59:59Z", 0, -86400, "AAA";
try "2006-01-02T00:00:00Z", 0, -86400, "AAA";
try "2006-01-02T23:59:59Z", 0, -86400, "AAA";

# changes occur in different order from how it superficially appears
$tz = DateTime::TimeZone::SystemV->new("AAA24BBB-24,J65/12,J66/12");
try "2005-01-31T00:00:00Z", 1, +86400, "BBB";
try "2005-03-04T00:00:00Z", 1, +86400, "BBB";
try "2005-03-05T00:00:00Z", 1, +86400, "BBB";
try "2005-03-06T00:00:00Z", 1, +86400, "BBB";
try "2005-03-06T11:59:59Z", 1, +86400, "BBB";
try "2005-03-06T12:00:00Z", 0, -86400, "AAA";
try "2005-03-06T23:59:59Z", 0, -86400, "AAA";
try "2005-03-07T00:00:00Z", 0, -86400, "AAA";
try "2005-03-07T11:59:59Z", 0, -86400, "AAA";
try "2005-03-07T12:00:00Z", 1, +86400, "BBB";
try "2005-03-07T23:59:59Z", 1, +86400, "BBB";
try "2005-03-08T00:00:00Z", 1, +86400, "BBB";
try "2005-03-09T00:00:00Z", 1, +86400, "BBB";
try "2005-12-01T00:00:00Z", 1, +86400, "BBB";

# changes occur in unexpected order across a year end
$tz = DateTime::TimeZone::SystemV->new("AAA24BBB-24,J365/12,J1/12");
try "2005-12-01T00:00:00Z", 1, +86400, "BBB";
try "2005-12-29T00:00:00Z", 1, +86400, "BBB";
try "2005-12-30T00:00:00Z", 1, +86400, "BBB";
try "2005-12-31T00:00:00Z", 1, +86400, "BBB";
try "2005-12-31T11:59:59Z", 1, +86400, "BBB";
try "2005-12-31T12:00:00Z", 0, -86400, "AAA";
try "2005-12-31T23:59:59Z", 0, -86400, "AAA";
try "2006-01-01T00:00:00Z", 0, -86400, "AAA";
try "2006-01-01T11:59:59Z", 0, -86400, "AAA";
try "2006-01-01T12:00:00Z", 1, +86400, "BBB";
try "2006-01-01T23:59:59Z", 1, +86400, "BBB";
try "2006-01-02T00:00:00Z", 1, +86400, "BBB";
try "2006-01-03T00:00:00Z", 1, +86400, "BBB";
try "2006-01-31T00:00:00Z", 1, +86400, "BBB";

# leap seconds, including changes occurring immediately after a leap second
# (these are not real leap second dates)
$tz = DateTime::TimeZone::SystemV->new("AAA-3BBB,M3.2.3/3,M11.1.6/4");
try "2005-02-01T12:00:00Z", 0, +10800, "AAA";
try "2005-02-01T23:59:59Z", 0, +10800, "AAA";
try "2005-02-01T23:59:60Z", 0, +10800, "AAA";
try "2005-02-02T00:00:00Z", 0, +10800, "AAA";
try "2005-02-02T12:00:00Z", 0, +10800, "AAA";
try "2005-03-08T12:00:00Z", 0, +10800, "AAA";
try "2005-03-08T23:59:59Z", 0, +10800, "AAA";
try "2005-03-08T23:59:60Z", 0, +10800, "AAA";
try "2005-03-09T00:00:00Z", 1, +14400, "BBB";
try "2005-03-09T12:00:00Z", 1, +14400, "BBB";
try "2005-04-01T12:00:00Z", 1, +14400, "BBB";
try "2005-04-01T23:59:59Z", 1, +14400, "BBB";
try "2005-04-01T23:59:60Z", 1, +14400, "BBB";
try "2005-04-02T00:00:00Z", 1, +14400, "BBB";
try "2005-04-02T12:00:00Z", 1, +14400, "BBB";
try "2005-11-04T12:00:00Z", 1, +14400, "BBB";
try "2005-11-04T23:59:59Z", 1, +14400, "BBB";
try "2005-11-04T23:59:60Z", 1, +14400, "BBB";
try "2005-11-05T00:00:00Z", 0, +10800, "AAA";
try "2005-11-05T12:00:00Z", 0, +10800, "AAA";
try "2005-12-01T12:00:00Z", 0, +10800, "AAA";
try "2005-12-01T23:59:59Z", 0, +10800, "AAA";
try "2005-12-01T23:59:60Z", 0, +10800, "AAA";
try "2005-12-02T00:00:00Z", 0, +10800, "AAA";
try "2005-12-02T12:00:00Z", 0, +10800, "AAA";

1;
