﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/connect/ConnectRequest.h>
#include <aws/connect/Connect_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Connect {
namespace Model {

/**
 */
class AssociateAnalyticsDataSetRequest : public ConnectRequest {
 public:
  AWS_CONNECT_API AssociateAnalyticsDataSetRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AssociateAnalyticsDataSet"; }

  AWS_CONNECT_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The identifier of the Amazon Connect instance. You can <a
   * href="https://docs.aws.amazon.com/connect/latest/adminguide/find-instance-arn.html">find
   * the instance ID</a> in the Amazon Resource Name (ARN) of the instance.</p>
   */
  inline const Aws::String& GetInstanceId() const { return m_instanceId; }
  inline bool InstanceIdHasBeenSet() const { return m_instanceIdHasBeenSet; }
  template <typename InstanceIdT = Aws::String>
  void SetInstanceId(InstanceIdT&& value) {
    m_instanceIdHasBeenSet = true;
    m_instanceId = std::forward<InstanceIdT>(value);
  }
  template <typename InstanceIdT = Aws::String>
  AssociateAnalyticsDataSetRequest& WithInstanceId(InstanceIdT&& value) {
    SetInstanceId(std::forward<InstanceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the dataset to associate with the target account.</p>
   */
  inline const Aws::String& GetDataSetId() const { return m_dataSetId; }
  inline bool DataSetIdHasBeenSet() const { return m_dataSetIdHasBeenSet; }
  template <typename DataSetIdT = Aws::String>
  void SetDataSetId(DataSetIdT&& value) {
    m_dataSetIdHasBeenSet = true;
    m_dataSetId = std::forward<DataSetIdT>(value);
  }
  template <typename DataSetIdT = Aws::String>
  AssociateAnalyticsDataSetRequest& WithDataSetId(DataSetIdT&& value) {
    SetDataSetId(std::forward<DataSetIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the target account. Use to associate a dataset to a
   * different account than the one containing the Amazon Connect instance. If not
   * specified, by default this value is the Amazon Web Services account that has the
   * Amazon Connect instance.</p>
   */
  inline const Aws::String& GetTargetAccountId() const { return m_targetAccountId; }
  inline bool TargetAccountIdHasBeenSet() const { return m_targetAccountIdHasBeenSet; }
  template <typename TargetAccountIdT = Aws::String>
  void SetTargetAccountId(TargetAccountIdT&& value) {
    m_targetAccountIdHasBeenSet = true;
    m_targetAccountId = std::forward<TargetAccountIdT>(value);
  }
  template <typename TargetAccountIdT = Aws::String>
  AssociateAnalyticsDataSetRequest& WithTargetAccountId(TargetAccountIdT&& value) {
    SetTargetAccountId(std::forward<TargetAccountIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_instanceId;

  Aws::String m_dataSetId;

  Aws::String m_targetAccountId;
  bool m_instanceIdHasBeenSet = false;
  bool m_dataSetIdHasBeenSet = false;
  bool m_targetAccountIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace Connect
}  // namespace Aws
