use strict;
use warnings;

use Test::More;

plan skip_all => "JSON module not installed" unless eval { require JSON };
plan tests => 91;

use Crypt::PK::Ed25519;
use Crypt::Misc qw(read_rawfile);

{
  my ($k, $k2);

  # t/data/openssl_ed25519_sk.pem
  # ED25519 Private-Key:
  # priv = 45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD == RcEJum_STotn0j77a5LZnNRX4hNxcsDXSf4rWgwULa0
  # pub  = A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D == oF0a6lgwrJplzfs4RmDUl-NpfEa0Gc8s7IXei9JFRZ0

  my $sk_data = pack("H*", "45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD");
  $k = Crypt::PK::Ed25519->new->import_key_raw($sk_data, 'private');
  ok($k, 'new+import_key_raw raw-priv');
  ok($k->is_private, 'is_private raw-priv');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} raw-priv');
  is(uc($k->key2hash->{pub}),  'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} raw-priv');
  is($k->export_key_raw('private'), $sk_data, 'export_key_raw private');

  $k2 = Crypt::PK::Ed25519->new->import_key($k->key2hash);
  ok($k2->is_private, 'is_private raw-priv');
  is($k->export_key_der('private'), $k2->export_key_der('private'), 'import_key hash');

  my $pk_data = pack("H*", "A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D");
  $k = Crypt::PK::Ed25519->new->import_key_raw($pk_data, 'public');
  ok($k, 'new+import_key_raw raw-pub');
  ok(!$k->is_private, '!is_private raw-pub');
  is(uc($k->key2hash->{pub}),  'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} raw-pub');
  is($k->export_key_raw('public'), $pk_data, 'export_key_raw public');

  $k2 = Crypt::PK::Ed25519->new->import_key($k->key2hash);
  ok(!$k2->is_private, 'is_private raw-priv');
  is($k->export_key_der('public'), $k2->export_key_der('public'), 'import_key hash');

  my $sk_jwk = { kty=>"OKP",crv=>"Ed25519",d=>"RcEJum_STotn0j77a5LZnNRX4hNxcsDXSf4rWgwULa0",x=>"oF0a6lgwrJplzfs4RmDUl-NpfEa0Gc8s7IXei9JFRZ0" };
  $k = Crypt::PK::Ed25519->new($sk_jwk);
  ok($k, 'new JWKHASH/priv');
  ok($k->is_private, 'is_private JWKHASH/priv');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} JWKHASH/priv');
  ok(eq_hash($sk_jwk, $k->export_key_jwk('private', 1)), 'JWKHASH export private');

  my $pk_jwk = { kty=>"OKP",crv=>"Ed25519",x=>"oF0a6lgwrJplzfs4RmDUl-NpfEa0Gc8s7IXei9JFRZ0" };
  $k = Crypt::PK::Ed25519->new($pk_jwk);
  ok($k, 'new JWKHASH/pub');
  ok(!$k->is_private, '!is_private JWKHASH/pub');
  is(uc($k->key2hash->{pub}), 'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} JWKHASH/pub');
  ok(eq_hash($pk_jwk, $k->export_key_jwk('public', 1)), 'JWKHASH export public');

  $k = Crypt::PK::Ed25519->new('t/data/jwk_ed25519-priv1.json');
  ok($k, 'new JWK/priv');
  ok($k->is_private, 'is_private JWK/priv');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} JWK/priv');

  $k = Crypt::PK::Ed25519->new('t/data/jwk_ed25519-pub1.json');
  ok($k, 'new JWK/pub');
  ok(!$k->is_private, '!is_private JWK/pub');
  is(uc($k->key2hash->{pub}), 'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} JWK/pub');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk.der');
  ok($k, 'new openssl_ed25519_sk.der');
  ok($k->is_private, 'is_private openssl_ed25519_sk.der');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk.der');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk.pem');
  ok($k, 'new openssl_ed25519_sk.pem');
  ok($k->is_private, 'is_private openssl_ed25519_sk.pem');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk_t.pem');
  ok($k, 'new openssl_ed25519_sk_t.pem');
  ok($k->is_private, 'is_private openssl_ed25519_sk_t.pem');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk_t.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk.pkcs8');
  ok($k, 'new openssl_ed25519_sk.pkcs8');
  ok($k->is_private, 'is_private openssl_ed25519_sk.pkcs8');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk.pkcs8');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk_pbes1.pkcs8', 'secret');
  ok($k, 'new openssl_ed25519_sk_pbes1.pkcs8');
  ok($k->is_private, 'is_private openssl_ed25519_sk_pbes1.pkcs8');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk_pbes1.pkcs8');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk_pbes2.pkcs8', 'secret');
  ok($k, 'new openssl_ed25519_sk_pbes2.pkcs8');
  ok($k->is_private, 'is_private openssl_ed25519_sk_pbes2.pkcs8');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk_pbes2.pkcs8');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk_pw.pem', 'secret');
  ok($k, 'new openssl_ed25519_sk_pw.pem');
  ok($k->is_private, 'is_private openssl_ed25519_sk_pw.pem');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk_pw.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_sk_pw_t.pem', 'secret');
  ok($k, 'new openssl_ed25519_sk_pw_t.pem');
  ok($k->is_private, 'is_private openssl_ed25519_sk_pw_t.pem');
  is(uc($k->key2hash->{priv}), '45C109BA6FD24E8B67D23EFB6B92D99CD457E2137172C0D749FE2B5A0C142DAD', 'key2hash->{priv} openssl_ed25519_sk_pw_t.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_pk.pem');
  ok($k, 'new openssl_ed25519_pk.pem');
  ok(!$k->is_private, '!is_private openssl_ed25519_pk.pem');
  is(uc($k->key2hash->{pub}), 'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} openssl_ed25519_pk.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_x509.pem');
  ok($k, 'new openssl_ed25519_x509.pem');
  ok(!$k->is_private, '!is_private openssl_ed25519_x509.pem');
  is(uc($k->key2hash->{pub}), 'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} openssl_ed25519_x509.pem');

  $k = Crypt::PK::Ed25519->new('t/data/openssl_ed25519_x509.der');
  ok($k, 'new openssl_ed25519_x509.der');
  ok(!$k->is_private, '!is_private openssl_ed25519_x509.der');
  is(uc($k->key2hash->{pub}), 'A05D1AEA5830AC9A65CDFB384660D497E3697C46B419CF2CEC85DE8BD245459D', 'key2hash->{pub} openssl_ed25519_x509.der');

  $k = Crypt::PK::Ed25519->new('t/data/ssh/ssh_ed25519.pub');
  ok($k, 'new ssh_ed25519.pub');
  ok(!$k->is_private, '!is_private ssh_ed25519.pub');
  is(uc($k->key2hash->{pub}), 'BD17B2215C443A7A1E9B286A4F0E76288130984CD942ACCCD4F1A064BB749FBE', 'key2hash->{pub} ssh_ed25519.pub');

  $k = Crypt::PK::Ed25519->new('t/data/ssh/ssh_ed25519.pub.rfc4716');
  ok($k, 'new ssh_ed25519.pub.rfc4716');
  ok(!$k->is_private, '!is_private ssh_ed25519.pub.rfc4716');
  is(uc($k->key2hash->{pub}), 'BD17B2215C443A7A1E9B286A4F0E76288130984CD942ACCCD4F1A064BB749FBE', 'key2hash->{pub} ssh_ed25519.pub.rfc4716');

  ### $k = Crypt::PK::Ed25519->new('t/data/ssh/ssh_ed25519.priv');
  ### ok($k, 'new ssh_ed25519.priv');
  ### ok($k->is_private, 'is_private ssh_ed25519.priv');

  ### $k = Crypt::PK::Ed25519->new('t/data/ssh/ssh_ed25519_pw.priv', 'secret');
  ### ok($k, 'new ssh_ed25519_pw.priv');
  ### ok($k->is_private, 'is_private ssh_ed25519_pw.priv');
}

{
  my $k = Crypt::PK::Ed25519->new;
  $k->generate_key;
  ok($k, 'generate_key');
  ok($k->is_private, 'is_private');
  ok($k->export_key_der('private'), 'export_key_der pri');
  ok($k->export_key_der('public'), 'export_key_der pub');
}

{
  for (qw( openssl_ed25519_pk.der openssl_ed25519_pk.pem )) {
    my $k = Crypt::PK::Ed25519->new("t/data/$_");
    is($k->export_key_der('public'), read_rawfile("t/data/$_"), 'export_key_der public') if (substr($_, -3) eq "der");
    is($k->export_key_pem('public'), read_rawfile("t/data/$_"), 'export_key_pem public') if (substr($_, -3) eq "pem");
  }

  for (qw( openssl_ed25519_sk.der openssl_ed25519_sk_t.pem )) {
    my $k = Crypt::PK::Ed25519->new("t/data/$_");
    is($k->export_key_der('private'), read_rawfile("t/data/$_"), 'export_key_der private') if (substr($_, -3) eq "der");
    is($k->export_key_pem('private'), read_rawfile("t/data/$_"), 'export_key_pem private') if (substr($_, -3) eq "pem");
  }
}

{
  my $sk = Crypt::PK::Ed25519->new;
  $sk->import_key('t/data/openssl_ed25519_sk.der');
  my $pk = Crypt::PK::Ed25519->new;
  $pk->import_key('t/data/openssl_ed25519_pk.der');

  my $sig = $sk->sign_message("message");
  ok(length $sig > 60, 'sign_message ' . length($sig));
  ok($pk->verify_message($sig, "message"), 'verify_message');
}

# test vectors from RFC 8032
{
  my @test_vectors = (
    {
      SECRETKEY => "9d61b19deffd5a60ba844af492ec2cc44449c5697b326919703bac031cae7f60",
      PUBLICKEY => "d75a980182b10ab7d54bfed3c964073a0ee172f3daa62325af021a68f707511a",
      MESSAGE   => "", # 0 bytes
      SIGNATURE => "e5564300c360ac729086e2cc806e828a84877f1eb8e5d974d873e065224901555fb8821590a33bacc61e39701cf9b46bd25bf5f0595bbe24655141438e7a100b",
    },
    {
      SECRETKEY => "4ccd089b28ff96da9db6c346ec114e0f5b8a319f35aba624da8cf6ed4fb8a6fb",
      PUBLICKEY => "3d4017c3e843895a92b70aa74d1b7ebc9c982ccf2ec4968cc0cd55f12af4660c",
      MESSAGE   => "72",
      SIGNATURE => "92a009a9f0d4cab8720e820b5f642540a2b27b5416503f8fb3762223ebdb69da085ac1e43e15996e458f3613d0f11d8c387b2eaeb4302aeeb00d291612bb0c00",
    },
    {
      SECRETKEY => "c5aa8df43f9f837bedb7442f31dcb7b166d38535076f094b85ce3a2e0b4458f7",
      PUBLICKEY => "fc51cd8e6218a1a38da47ed00230f0580816ed13ba3303ac5deb911548908025",
      MESSAGE   => "af82",
      SIGNATURE => "6291d657deec24024827e69c3abe01a30ce548a284743a445e3680d7db5ac3ac18ff9b538d16f290ae67f760984dc6594a7c15e9716ed28dc027beceea1ec40a",
    },
    {
      SECRETKEY => "f5e5767cf153319517630f226876b86c8160cc583bc013744c6bf255f5cc0ee5",
      PUBLICKEY => "278117fc144c72340f67d0f2316e8386ceffbf2b2428c9c51fef7c597f1d426e",
      MESSAGE   => "08b8b2b733424243760fe426a4b54908632110a66c2f6591eabd3345e3e4eb98fa6e264bf09efe12ee50f8f54e9f77b1e355f6c50544e23fb1433ddf73be84d8".
                   "79de7c0046dc4996d9e773f4bc9efe5738829adb26c81b37c93a1b270b20329d658675fc6ea534e0810a4432826bf58c941efb65d57a338bbd2e26640f89ffbc".
                   "1a858efcb8550ee3a5e1998bd177e93a7363c344fe6b199ee5d02e82d522c4feba15452f80288a821a579116ec6dad2b3b310da903401aa62100ab5d1a36553e".
                   "06203b33890cc9b832f79ef80560ccb9a39ce767967ed628c6ad573cb116dbefefd75499da96bd68a8a97b928a8bbc103b6621fcde2beca1231d206be6cd9ec7".
                   "aff6f6c94fcd7204ed3455c68c83f4a41da4af2b74ef5c53f1d8ac70bdcb7ed185ce81bd84359d44254d95629e9855a94a7c1958d1f8ada5d0532ed8a5aa3fb2".
                   "d17ba70eb6248e594e1a2297acbbb39d502f1a8c6eb6f1ce22b3de1a1f40cc24554119a831a9aad6079cad88425de6bde1a9187ebb6092cf67bf2b13fd65f270".
                   "88d78b7e883c8759d2c4f5c65adb7553878ad575f9fad878e80a0c9ba63bcbcc2732e69485bbc9c90bfbd62481d9089beccf80cfe2df16a2cf65bd92dd597b07".
                   "07e0917af48bbb75fed413d238f5555a7a569d80c3414a8d0859dc65a46128bab27af87a71314f318c782b23ebfe808b82b0ce26401d2e22f04d83d1255dc51a".
                   "ddd3b75a2b1ae0784504df543af8969be3ea7082ff7fc9888c144da2af58429ec96031dbcad3dad9af0dcbaaaf268cb8fcffead94f3c7ca495e056a9b47acdb7".
                   "51fb73e666c6c655ade8297297d07ad1ba5e43f1bca32301651339e22904cc8c42f58c30c04aafdb038dda0847dd988dcda6f3bfd15c4b4c4525004aa06eeff8".
                   "ca61783aacec57fb3d1f92b0fe2fd1a85f6724517b65e614ad6808d6f6ee34dff7310fdc82aebfd904b01e1dc54b2927094b2db68d6f903b68401adebf5a7e08".
                   "d78ff4ef5d63653a65040cf9bfd4aca7984a74d37145986780fc0b16ac451649de6188a7dbdf191f64b5fc5e2ab47b57f7f7276cd419c17a3ca8e1b939ae49e4".
                   "88acba6b965610b5480109c8b17b80e1b7b750dfc7598d5d5011fd2dcc5600a32ef5b52a1ecc820e308aa342721aac0943bf6686b64b2579376504ccc493d97e".
                   "6aed3fb0f9cd71a43dd497f01f17c0e2cb3797aa2a2f256656168e6c496afc5fb93246f6b1116398a346f1a641f3b041e989f7914f90cc2c7fff357876e506b5".
                   "0d334ba77c225bc307ba537152f3f1610e4eafe595f6d9d90d11faa933a15ef1369546868a7f3a45a96768d40fd9d03412c091c6315cf4fde7cb68606937380d".
                   "b2eaaa707b4c4185c32eddcdd306705e4dc1ffc872eeee475a64dfac86aba41c0618983f8741c5ef68d3a101e8a3b8cac60c905c15fc910840b94c00a0b9d0",
      SIGNATURE => "0aab4c900501b3e24d7cdf4663326a3a87df5e4843b2cbdb67cbf6e460fec350aa5371b1508f9f4528ecea23c436d94b5e8fcd4f681e30a6ac00a9704a188a03",
    },
    {
      SECRETKEY => "833fe62409237b9d62ec77587520911e9a759cec1d19755b7da901b96dca3d42",
      PUBLICKEY => "ec172b93ad5e563bf4932c70e1245034c35467ef2efd4d64ebf819683467e2bf",
      MESSAGE   => "ddaf35a193617abacc417349ae20413112e6fa4e89a97ea20a9eeee64b55d39a2192992a274fc1a836ba3c23a3feebbd454d4423643ce80e2a9ac94fa54ca49f",
      SIGNATURE => "dc2a4459e7369633a52b1bf277839a00201009a3efbf3ecb69bea2186c26b58909351fc9ac90b3ecfdfbc7c66431e0303dca179c138ac17ad9bef1177331a704",
    }
  );
  for my $tv (@test_vectors) {
    my $sk  = pack("H*", $tv->{SECRETKEY});
    my $pk  = pack("H*", $tv->{PUBLICKEY});
    my $msg = pack("H*", $tv->{MESSAGE});
    my $sig = pack("H*", $tv->{SIGNATURE});
    my $s = Crypt::PK::Ed25519->new->import_key_raw($sk, 'private');
    my $p = Crypt::PK::Ed25519->new->import_key_raw($pk, 'public');
    my $sigx = $s->sign_message($msg);
    my $pkx  = $s->export_key_raw('public');
    is(unpack("H*", $sigx), $tv->{SIGNATURE}, 'sign');
    is(unpack("H*", $pkx), $tv->{PUBLICKEY}, 'public key');
    ok($p->verify_message($sig, $msg), 'verify');
  }
}
