/*
 * unity-webapps-notification-context.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>

#include "unity-webapps-notification-context.h"
#include "unity-webapps-context-private.h"

#include "unity-webapps-sanitizer.h"

#include "unity-webapps-dbus-util.h"

#include "unity-webapps-rate.h"

#include "unity-webapps-debug.h"

UnityWebappsNotificationContext *
unity_webapps_notification_context_new (UnityWebappsContext *main_context, GError **error)
{
  UnityWebappsNotificationContext *context;
  
  context = g_malloc0 (sizeof (UnityWebappsNotificationContext));
  context->notification_rate = 0;
  
  context->notification_proxy = 
    unity_webapps_gen_notification_proxy_new_sync (unity_webapps_service_get_connection (main_context->priv->service) ,
						   G_DBUS_PROXY_FLAGS_NONE,
						   main_context->priv->context_name,
						   UNITY_WEBAPPS_NOTIFICATION_PATH, 
						   NULL /* Cancellable */,
						   error);
  
  if (error && (*error != NULL))
    {
      g_critical ("Error creating notification context proxy object for %s: %s", main_context->priv->context_name, (*error)->message);
      
      return NULL;
    }
  
  return context;
}

void 
unity_webapps_notification_context_free (UnityWebappsNotificationContext *context)
{
  g_return_if_fail (context != NULL);
  g_object_unref (G_OBJECT (context->notification_proxy));
  
  g_free (context);
}

UNITY_WEBAPPS_DEFINE_VOID_DBUS_HANDLER(show_notification,notification,Notification,NOTIFICATION,"Notification succesfully sent");

#define MAXIMUM_SUMMARY_LENGTH 80
#define MAXIMUM_BODY_LENGTH 160
#define MAXIMUM_ICON_URL_LENGTH 4096

void
unity_webapps_notification_show_notification (UnityWebappsContext *context,
					      const gchar *summary,
					      const gchar *body,
					      const gchar *icon_url)
{
  gchar *sanitized_summary, *sanitized_body, *sanitized_icon_url;
  g_return_if_fail (context != NULL);
  g_return_if_fail (UNITY_WEBAPPS_IS_CONTEXT (context));
  g_return_if_fail (summary != NULL);
  g_return_if_fail (body != NULL);
  
  if (unity_webapps_rate_check_notification_rate_limit (context) == FALSE)
    {
      return;
    }
  
  sanitized_summary = unity_webapps_sanitizer_limit_string_argument (summary, MAXIMUM_SUMMARY_LENGTH);
  sanitized_body = unity_webapps_sanitizer_limit_string_argument (body, MAXIMUM_BODY_LENGTH);
  sanitized_icon_url = unity_webapps_sanitizer_limit_string_argument (icon_url, MAXIMUM_ICON_URL_LENGTH);
  
  
  unity_webapps_gen_notification_call_show_notification (context->priv->notification_context->notification_proxy,
							 sanitized_summary,
							 sanitized_body,
							 sanitized_icon_url,
							      NULL /* Cancellable */,
							 show_notification_complete_callback,
							 context);
  
  g_free (sanitized_summary);
  g_free (sanitized_body);
  g_free (sanitized_icon_url);
  
  UNITY_WEBAPPS_NOTE (NOTIFICATION, "Showed notification, summary: %s", summary);
}
